// ****************************************************************************
// 
// FLV Input Driver Plugin for VirtualDub
// Copyright (C) 2007  Moitah (moitah@yahoo.com)
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

#define _CRT_SECURE_NO_WARNINGS
#include <vd2/plugin/vdplugin.h>
#include <vd2/plugin/vdinputdriver.h>
#include <vector>
#include <math.h>
#include <io.h>
#include <windows.h>
#include <vfw.h>

///////////////////////////////////////////////////////////////////////////////

extern "C" long _InterlockedExchangeAdd(volatile long *p, long v);
#pragma intrinsic(_InterlockedExchangeAdd)

namespace {
	template<class T> class vdxunknown : public T {
	public:
		vdxunknown() : mRefCount(0) {}
		vdxunknown(const vdxunknown<T>& src) : mRefCount(0) {}		// do not copy the refcount
		virtual ~vdxunknown() {}

		vdxunknown<T>& operator=(const vdxunknown<T>&) {}			// do not copy the refcount

		virtual int VDXAPIENTRY AddRef() {
			return _InterlockedExchangeAdd(&mRefCount, 1) + 1;
		}

		virtual int VDXAPIENTRY Release() {
			long rc = _InterlockedExchangeAdd(&mRefCount, -1) - 1;
			if (!mRefCount) {
				mRefCount = 1;
				delete this;
				return 0;
			}

			return rc;
		}

		virtual void *VDXAPIENTRY AsInterface(uint32 iid) {
			if (iid == T::kIID)
				return static_cast<T *>(this);

			if (iid == IVDXUnknown::kIID)
				return static_cast<IVDXUnknown *>(this);

			return NULL;
		}

	protected:
		volatile long	mRefCount;
	};
}

///////////////////////////////////////////////////////////////////////////////

enum FLVFourCC : DWORD {
	kFCC_FLV1 = MAKEFOURCC('F', 'L', 'V', '1'),
	kFCC_VP6F = MAKEFOURCC('V', 'P', '6', 'F'),
};

enum FLVCodec {
	kVC_None,
	kVC_H263,
	kVC_VP6,
	kVC_VP6A,
	kVC_Screen,
	kVC_Screen2,
	kAC_None,
	kAC_PCM,
	kAC_ADPCM,
	kAC_MP3,
	kAC_Nellymoser,
	kAC_Nellymoser8M,
};

struct FLVTagInfo {
	sint64	mDataOffset;
	uint32	mDataSize;   // Most significant byte contains flags: 0x01 = Keyframe
	uint32	mTimeStamp;  // In milliseconds
};

class FLVData {
public:
	FILE *mFile;
	int mVideoCodec;
	BITMAPINFOHEADER mVideoFormat;
	std::vector<FLVTagInfo> mVideoIndex;
	int mAudioCodec;
	std::vector<FLVTagInfo> mAudioIndex;
	MPEGLAYER3WAVEFORMAT mAudioFormat;   // Used for PCM too (only the WAVEFORMATEX part)
	sint64 mAudioTotalBytes;
	uint32 mAudioEstimatedDuration;
};

namespace {
	uint32 GetInt32(uint8 *&buff) {
		uint32 x = ((uint32)buff[0] << 24) |
				   ((uint32)buff[1] << 16) |
				   ((uint32)buff[2] <<  8) |
				   ((uint32)buff[3]      );
		buff += 4;
		return x;
	}

	uint32 GetInt24(uint8 *&buff) {
		uint32 x = ((uint32)buff[0] << 16) |
				   ((uint32)buff[1] <<  8) |
				   ((uint32)buff[2]      );
		buff += 3;
		return x;
	}

	uint32 GetInt16(uint8 *&buff) {
		uint32 x = ((uint32)buff[0] <<  8) |
				   ((uint32)buff[1]      );
		buff += 2;
		return x;
	}

	uint32 GetInt8(uint8 *&buff) {
		return (uint32)*(buff++);
	}

	uint64 ToBits(uint8 *buff, sint32 buffLen) {
		uint64 bits = 0;
		uint8 *pBits = (uint8*)&bits;
		int i = 7;

		while ((i > 0) && (buffLen > 0)) {
			pBits[i--] = *(buff++);
			buffLen--;
		}

		return bits;
	}

	uint32 GetBits(uint64 &bits, int len) {
		uint32 x = (uint32)(bits >> (64 - len));
		bits <<= len;
		return x;
	}

	void GetFrameSize(uint8 *buff, sint32 buffLen, int codec, sint32 &width, sint32 &height) {
		if (codec == kVC_H263) {
			uint64 bits = ToBits(buff + 2, buffLen - 2);
			uint32 format;

			if (GetBits(bits, 1) != 1) return;
			GetBits(bits, 5);
			GetBits(bits, 8);
			format = GetBits(bits, 3);

			switch (format) {
				case 0:
					width = (sint32)GetBits(bits, 8);
					height = (sint32)GetBits(bits, 8);
					break;
				case 1:
					width = (sint32)GetBits(bits, 16);
					height = (sint32)GetBits(bits, 16);
					break;
				case 2: width = 352; height = 288; break;
				case 3: width = 176; height = 144; break;
				case 4: width = 128; height =  96; break;
				case 5: width = 320; height = 240; break;
				case 6: width = 160; height = 120; break;
			}
		}
		else if ((codec == kVC_VP6) || (codec == kVC_VP6A)) {
			uint64 bits = ToBits(buff, buffLen);
			uint32 separatedCoeffFlag, filterHeader;

			if (GetBits(bits, 1)) return; // Delta frame
			GetBits(bits, 6);
			separatedCoeffFlag = GetBits(bits, 1);
			GetBits(bits, 5);
			filterHeader = GetBits(bits, 2);
			GetBits(bits, 1);
			if ((separatedCoeffFlag != 0) || (filterHeader == 0)) {
				GetBits(bits, 16);
			}

			height = (sint32)GetBits(bits, 8) * 16;
			width = (sint32)GetBits(bits, 8) * 16;
		}
	}

	sint64 TotalStreamBytes(std::vector<FLVTagInfo> tagList) {
		sint64 totalBytes = 0;
		for (int i = 0; i < tagList.size(); i++) {
			totalBytes += tagList[i].mDataSize & 0x00FFFFFF;
		}
		return totalBytes;
	}
}

///////////////////////////////////////////////////////////////////////////////

class VideoDecoderFLV : public vdxunknown<IVDXVideoDecoder> {

private:
	HIC hCodec;
	bool QueryDecompressor(BITMAPINFOHEADER *out);

public:
	VideoDecoderFLV(FLVData& data);
	~VideoDecoderFLV();

	const void *VDXAPIENTRY DecodeFrame(const void *inputBuffer, uint32 data_len, bool is_preroll, sint64 streamFrame, sint64 targetFrame);
	uint32		VDXAPIENTRY GetDecodePadding();
	void		VDXAPIENTRY Reset();
	bool		VDXAPIENTRY IsFrameBufferValid();
	const VDXPixmap& VDXAPIENTRY GetFrameBuffer();
	bool		VDXAPIENTRY SetTargetFormat(int format, bool useDIBAlignment);
	bool		VDXAPIENTRY SetDecompressedFormat(const VDXBITMAPINFOHEADER *pbih);

	const void *VDXAPIENTRY GetFrameBufferBase();
	bool		VDXAPIENTRY IsDecodable(sint64 sample_num);

protected:
	VDXPixmap	mPixmap;
	BITMAPINFOHEADER bmihIn;
	BITMAPINFOHEADER bmihOut;
	unsigned char *mFrameBuffer;

	sint64 mPrevFrame;

	FLVData& mData;
};

VideoDecoderFLV::VideoDecoderFLV(FLVData& data)
	: mData(data)
{
	hCodec = NULL;
	mPrevFrame = -1;
	bmihIn = mData.mVideoFormat;

	// Default
	bmihOut.biSize = sizeof(bmihOut);
	bmihOut.biWidth = bmihIn.biWidth;
	bmihOut.biHeight = bmihIn.biHeight;
	bmihOut.biPlanes = 1;
	bmihOut.biBitCount = 24;
	bmihOut.biCompression = BI_RGB;
	bmihOut.biSizeImage = 0;
	bmihOut.biXPelsPerMeter = 0;
	bmihOut.biYPelsPerMeter = 0;
	bmihOut.biClrUsed = 0;
	bmihOut.biClrImportant = 0;

	mFrameBuffer = (unsigned char *)malloc(bmihIn.biWidth * bmihIn.biHeight * 4);
	if (mFrameBuffer != NULL) {
		memset(mFrameBuffer, 0, bmihIn.biWidth * bmihIn.biHeight * 4);
	}

	mPixmap.pitch		= (bmihIn.biWidth * 3 + 3) & 0xFFFFFFFC;
	mPixmap.data		= mFrameBuffer + (bmihIn.biHeight - 1) * mPixmap.pitch;
	mPixmap.pitch		= -mPixmap.pitch;
	mPixmap.palette		= NULL;
	mPixmap.format		= nsVDXPixmap::kPixFormat_RGB888;
	mPixmap.w			= bmihIn.biWidth;
	mPixmap.h			= bmihIn.biHeight;
	mPixmap.data2		= NULL;
	mPixmap.pitch2		= 0;
	mPixmap.data3		= NULL;
	mPixmap.pitch3		= 0;
}

VideoDecoderFLV::~VideoDecoderFLV() {
	if (hCodec != NULL) {
		ICDecompressEnd(hCodec);
		ICClose(hCodec);
	}
	free(mFrameBuffer);
}

const void *VideoDecoderFLV::DecodeFrame(
	const void *inputBuffer, uint32 data_len, bool is_preroll,
	sint64 streamFrame,	sint64 targetFrame)
{
	if ((hCodec != NULL) && (inputBuffer != NULL) && (streamFrame != -1)) {
		DWORD f;

		f = (mData.mVideoIndex[(sint32)streamFrame].mDataSize & 0xFF000000) ?
			0 : ICDECOMPRESS_NOTKEYFRAME;

		if (is_preroll) {
			f |= ICDECOMPRESS_PREROLL;
		} else {
			mPrevFrame = targetFrame;
		}

		bmihIn.biSizeImage = data_len;

		f = ICDecompress(hCodec, f, &bmihIn, (LPVOID)inputBuffer,
			&bmihOut, mFrameBuffer);
	}

	return mFrameBuffer;
}

uint32 VideoDecoderFLV::GetDecodePadding() {
	return 0;
}

void VideoDecoderFLV::Reset() {
	mPrevFrame = -1;
}

bool VideoDecoderFLV::IsFrameBufferValid() {
	return (mPrevFrame >= 0);
}

const VDXPixmap& VideoDecoderFLV::GetFrameBuffer() {
	return mPixmap;
}

bool VideoDecoderFLV::QueryDecompressor(BITMAPINFOHEADER *out) {
	bool ret;

	if (hCodec != NULL) {
		ret = (ICDecompressQuery(hCodec, &bmihIn, out) == ICERR_OK);
	} else {
		hCodec = ICLocate(ICTYPE_VIDEO, bmihIn.biCompression, &bmihIn, out, ICMODE_DECOMPRESS);
		ret = (hCodec != NULL);
	}

	if (ret) ICDecompressBegin(hCodec, &bmihIn, out);
	return ret;
}

bool VideoDecoderFLV::SetTargetFormat(int format, bool useDIBAlignment) {
	BITMAPINFOHEADER bmihTest;
	long w, h;

	w = bmihIn.biWidth;
	h = bmihIn.biHeight;

	using namespace nsVDXPixmap;

	if (format == kPixFormat_Null) format = kPixFormat_RGB888;

	bmihTest.biSize = sizeof(bmihTest);
	bmihTest.biWidth = w;
	bmihTest.biHeight = h;
	bmihTest.biSizeImage = 0;
	bmihTest.biXPelsPerMeter = 0;
	bmihTest.biYPelsPerMeter = 0;
	bmihTest.biClrUsed = 0;
	bmihTest.biClrImportant = 0;

	switch (format) {

	case kPixFormat_XRGB1555:
		bmihTest.biPlanes = 1;
		bmihTest.biBitCount = 16;
		bmihTest.biCompression = BI_RGB;

		if (!QueryDecompressor(&bmihTest)) return false;
		memcpy(&bmihOut, &bmihTest, sizeof(bmihTest));

		mPixmap.pitch		= (w * 2 + 3) & 0xFFFFFFFC;
		mPixmap.data		= mFrameBuffer + (h - 1) * mPixmap.pitch;
		mPixmap.pitch		= -mPixmap.pitch;
		mPixmap.palette		= NULL;
		mPixmap.format		= format;
		mPixmap.w			= w;
		mPixmap.h			= h;
		mPixmap.data2		= NULL;
		mPixmap.pitch2		= 0;
		mPixmap.data3		= NULL;
		mPixmap.pitch3		= 0;
		break;

	case kPixFormat_RGB888:
		bmihTest.biPlanes = 1;
		bmihTest.biBitCount = 24;
		bmihTest.biCompression = BI_RGB;

		if (!QueryDecompressor(&bmihTest)) return false;
		memcpy(&bmihOut, &bmihTest, sizeof(bmihTest));

		mPixmap.pitch		= (w * 3 + 3) & 0xFFFFFFFC;
		mPixmap.data		= mFrameBuffer + (h - 1) * mPixmap.pitch;
		mPixmap.pitch		= -mPixmap.pitch;
		mPixmap.palette		= NULL;
		mPixmap.format		= format;
		mPixmap.w			= w;
		mPixmap.h			= h;
		mPixmap.data2		= NULL;
		mPixmap.pitch2		= 0;
		mPixmap.data3		= NULL;
		mPixmap.pitch3		= 0;
		break;

	case kPixFormat_XRGB8888:
		bmihTest.biPlanes = 1;
		bmihTest.biBitCount = 32;
		bmihTest.biCompression = BI_RGB;

		if (!QueryDecompressor(&bmihTest)) return false;
		memcpy(&bmihOut, &bmihTest, sizeof(bmihTest));

		mPixmap.pitch		= w * 4;
		mPixmap.data		= mFrameBuffer + (h - 1) * mPixmap.pitch;
		mPixmap.pitch		= -mPixmap.pitch;
		mPixmap.palette		= NULL;
		mPixmap.format		= format;
		mPixmap.w			= w;
		mPixmap.h			= h;
		mPixmap.data2		= NULL;
		mPixmap.pitch2		= 0;
		mPixmap.data3		= NULL;
		mPixmap.pitch3		= 0;
		break;

	case kPixFormat_YUV422_UYVY:
		bmihTest.biCompression = 'YVYU';
		goto Check422;

	case kPixFormat_YUV422_YUYV:
		bmihTest.biCompression = '2YUY';

Check422:
		if (w & 1) return false;

		bmihTest.biPlanes = 1;
		bmihTest.biBitCount = 16;
		bmihTest.biSizeImage = ((w * 2 + 3) & 0xFFFFFFFC) * h;

		if (!QueryDecompressor(&bmihTest)) return false;
		memcpy(&bmihOut, &bmihTest, sizeof(bmihTest));

		mPixmap.data		= mFrameBuffer;
		mPixmap.pitch		= (w * 2 + 3) & 0xFFFFFFFC;
		mPixmap.palette		= NULL;
		mPixmap.format		= format;
		mPixmap.w			= w;
		mPixmap.h			= h;
		mPixmap.data2		= NULL;
		mPixmap.pitch2		= 0;
		mPixmap.data3		= NULL;
		mPixmap.pitch3		= 0;
		break;

	case kPixFormat_YUV420_Planar:
		if ((w | h) & 1) return false;

		bmihTest.biPlanes = 3;
		bmihTest.biBitCount = 12;
		bmihTest.biCompression = '21VY';
		bmihTest.biSizeImage = (w * h) + (w >> 1) * (h >> 1) * 2;

		if (!QueryDecompressor(&bmihTest)) return false;
		memcpy(&bmihOut, &bmihTest, sizeof(bmihTest));

		mPixmap.data		= mFrameBuffer;
		mPixmap.pitch		= w;
		mPixmap.palette		= NULL;
		mPixmap.format		= format;
		mPixmap.w			= w;
		mPixmap.h			= h;
		mPixmap.data2		= mFrameBuffer + w * h + (w >> 1) * (h >> 1);
		mPixmap.pitch2		= w >> 1;
		mPixmap.data3		= mFrameBuffer + w * h;
		mPixmap.pitch3		= w >> 1;
		break;

	default:
		return false;
	}

	Reset();

	return true;
}

bool VideoDecoderFLV::SetDecompressedFormat(const VDXBITMAPINFOHEADER *pbih) {
	return false;
}

const void *VideoDecoderFLV::GetFrameBufferBase() {
	return mFrameBuffer;
}

bool VideoDecoderFLV::IsDecodable(sint64 sample_num64) {
	if (mData.mVideoIndex[(uint32)sample_num64].mDataSize & 0xFF000000) return true;
	if ((sample_num64 - 1) == mPrevFrame) return true;
	return false;
}

///////////////////////////////////////////////////////////////////////////////

class VideoSourceFLV : public vdxunknown<IVDXStreamSource>, public IVDXVideoSource {
public:
	VideoSourceFLV(FLVData& data);
	~VideoSourceFLV();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();
	void *VDXAPIENTRY AsInterface(uint32 iid);

	void		VDXAPIENTRY GetStreamSourceInfo(VDXStreamSourceInfo&);
	bool		VDXAPIENTRY Read(sint64 lStart, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead);

	const void *VDXAPIENTRY GetDirectFormat();
	int			VDXAPIENTRY GetDirectFormatLen();

	ErrorMode VDXAPIENTRY GetDecodeErrorMode();
	void VDXAPIENTRY SetDecodeErrorMode(ErrorMode mode);
	bool VDXAPIENTRY IsDecodeErrorModeSupported(ErrorMode mode);

	bool VDXAPIENTRY IsVBR();
	sint64 VDXAPIENTRY TimeToPositionVBR(sint64 us);
	sint64 VDXAPIENTRY PositionToTimeVBR(sint64 samples);

	void VDXAPIENTRY GetVideoSourceInfo(VDXVideoSourceInfo& info);

	bool VDXAPIENTRY CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel);
	bool VDXAPIENTRY CreateVideoDecoder(IVDXVideoDecoder **ppDecoder);

	void		VDXAPIENTRY GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo);

	bool		VDXAPIENTRY IsKey(sint64 lSample);

	sint64		VDXAPIENTRY GetFrameNumberForSample(sint64 sample_num);
	sint64		VDXAPIENTRY GetSampleNumberForFrame(sint64 display_num);
	sint64		VDXAPIENTRY GetRealFrame(sint64 display_num);

	sint64		VDXAPIENTRY GetSampleBytePosition(sint64 sample_num);

protected:
	FLVData&	mData;
};

VideoSourceFLV::VideoSourceFLV(FLVData& data)
	: mData(data)
{
}

VideoSourceFLV::~VideoSourceFLV() {
}

int VideoSourceFLV::AddRef() {
	return vdxunknown<IVDXStreamSource>::AddRef();
}

int VideoSourceFLV::Release() {
	return vdxunknown<IVDXStreamSource>::Release();
}

void *VDXAPIENTRY VideoSourceFLV::AsInterface(uint32 iid) {
	if (iid == IVDXVideoSource::kIID)
		return static_cast<IVDXVideoSource *>(this);

	return vdxunknown<IVDXStreamSource>::AsInterface(iid);
}

void VDXAPIENTRY VideoSourceFLV::GetStreamSourceInfo(VDXStreamSourceInfo& srcInfo) {
	int frameCount = mData.mVideoIndex.size();
	srcInfo.mSampleRate.mNumerator = (frameCount - 1) * 1000;
	srcInfo.mSampleRate.mDenominator = mData.mVideoIndex[frameCount-1].mTimeStamp - mData.mVideoIndex[0].mTimeStamp;
	srcInfo.mSampleCount = frameCount;
}

bool VideoSourceFLV::Read(sint64 lStart64, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead) {
	const FLVTagInfo& tagInfo = mData.mVideoIndex[(uint32)lStart64];

	*lBytesRead = tagInfo.mDataSize & 0x00FFFFFF;
	*lSamplesRead = 1;

	if (lpBuffer) {
		if (cbBuffer < *lBytesRead)
			return false;

		_fseeki64(mData.mFile, tagInfo.mDataOffset, SEEK_SET);
		fread(lpBuffer, 1, *lBytesRead, mData.mFile);
	}

	return true;
}

const void *VideoSourceFLV::GetDirectFormat() {
	return &mData.mVideoFormat;
}

int VideoSourceFLV::GetDirectFormatLen() {
	return sizeof(mData.mVideoFormat);
}

IVDXStreamSource::ErrorMode VideoSourceFLV::GetDecodeErrorMode() {
	return IVDXStreamSource::kErrorModeReportAll;
}

void VideoSourceFLV::SetDecodeErrorMode(IVDXStreamSource::ErrorMode mode) {
}

bool VideoSourceFLV::IsDecodeErrorModeSupported(IVDXStreamSource::ErrorMode mode) {
	return mode == IVDXStreamSource::kErrorModeReportAll;
}

bool VideoSourceFLV::IsVBR() {
	return false;
}

sint64 VideoSourceFLV::TimeToPositionVBR(sint64 us) {
	return 0;
}

sint64 VideoSourceFLV::PositionToTimeVBR(sint64 samples) {
	return 0;
}

void VideoSourceFLV::GetVideoSourceInfo(VDXVideoSourceInfo& info) {
	info.mFlags = 0;
	info.mWidth = mData.mVideoFormat.biWidth;
	info.mHeight = mData.mVideoFormat.biHeight;
	info.mDecoderModel = VDXVideoSourceInfo::kDecoderModelDefaultIP;
}

bool VideoSourceFLV::CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel) {
	return false;
}

bool VideoSourceFLV::CreateVideoDecoder(IVDXVideoDecoder **ppDecoder) {
	VideoDecoderFLV *p = new VideoDecoderFLV(mData);
	if (!p)
		return false;
	p->AddRef();
	*ppDecoder = p;
	return true;
}

void VideoSourceFLV::GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo) {
	frameInfo.mBytePosition = -1;

	if (mData.mVideoIndex[(uint32)sample_num].mDataSize & 0xFF000000) {
		frameInfo.mFrameType = kVDXVFT_Independent;
		frameInfo.mTypeChar = 'K';
	}
	else {
		frameInfo.mFrameType = kVDXVFT_Predicted;
		frameInfo.mTypeChar = ' ';
	}
}

bool VideoSourceFLV::IsKey(sint64 sample) {
	return !!(mData.mVideoIndex[(uint32)sample].mDataSize & 0xFF000000);
}

sint64 VideoSourceFLV::GetFrameNumberForSample(sint64 sample_num) {
	return sample_num;
}

sint64 VideoSourceFLV::GetSampleNumberForFrame(sint64 display_num) {
	return display_num;
}

sint64 VideoSourceFLV::GetRealFrame(sint64 display_num) {
	return display_num;
}

sint64 VideoSourceFLV::GetSampleBytePosition(sint64 sample_num) {
	return -1;
}

///////////////////////////////////////////////////////////////////////////////

class AudioSourceFLV : public vdxunknown<IVDXStreamSource>, public IVDXAudioSource {
public:
	AudioSourceFLV(FLVData& data);
	~AudioSourceFLV();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();
	void *VDXAPIENTRY AsInterface(uint32 iid);

	void		VDXAPIENTRY GetStreamSourceInfo(VDXStreamSourceInfo&);
	bool		VDXAPIENTRY Read(sint64 lStart, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead);

	const void *VDXAPIENTRY GetDirectFormat();
	int			VDXAPIENTRY GetDirectFormatLen();

	ErrorMode VDXAPIENTRY GetDecodeErrorMode();
	void VDXAPIENTRY SetDecodeErrorMode(ErrorMode mode);
	bool VDXAPIENTRY IsDecodeErrorModeSupported(ErrorMode mode);

	bool VDXAPIENTRY IsVBR();
	sint64 VDXAPIENTRY TimeToPositionVBR(sint64 us);
	sint64 VDXAPIENTRY PositionToTimeVBR(sint64 samples);

	void VDXAPIENTRY GetAudioSourceInfo(VDXAudioSourceInfo& info);

protected:
	FLVData&	mData;
};

AudioSourceFLV::AudioSourceFLV(FLVData& data)
	: mData(data)
{
}

AudioSourceFLV::~AudioSourceFLV() {
}

int AudioSourceFLV::AddRef() {
	return vdxunknown<IVDXStreamSource>::AddRef();
}

int AudioSourceFLV::Release() {
	return vdxunknown<IVDXStreamSource>::Release();
}

void *VDXAPIENTRY AudioSourceFLV::AsInterface(uint32 iid) {
	if (iid == IVDXVideoSource::kIID)
		return static_cast<IVDXAudioSource *>(this);

	return vdxunknown<IVDXStreamSource>::AsInterface(iid);
}

void VDXAPIENTRY AudioSourceFLV::GetStreamSourceInfo(VDXStreamSourceInfo& srcInfo) {
	srcInfo.mSampleRate.mNumerator = mData.mAudioFormat.wfx.nAvgBytesPerSec;
	srcInfo.mSampleRate.mDenominator = mData.mAudioFormat.wfx.nBlockAlign;
	srcInfo.mSampleCount = mData.mAudioTotalBytes / mData.mAudioFormat.wfx.nBlockAlign;

	//char dbgBuff[256];
	//sprintf(dbgBuff, "Rate = %d / %d, count = %d, avg frame size = %d.", srcInfo.mSampleRate.mNumerator, 
	//	srcInfo.mSampleRate.mDenominator, (uint32)srcInfo.mSampleCount, mData.mAudioFormat.nBlockSize);
	//OutputDebugString(dbgBuff);
}

bool AudioSourceFLV::Read(sint64 lStart64, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead) {
	FLVTagInfo tagInfo;
	uint32 tagIndex, tagDataSize, bytesPerSample;
	sint64 tagDataSkip;

	bytesPerSample = mData.mAudioFormat.wfx.nBlockAlign;
	tagDataSkip = lStart64 * bytesPerSample;
	tagIndex = 0;
	while (true) {
		tagInfo = mData.mAudioIndex[tagIndex];
		tagDataSize = tagInfo.mDataSize & 0x00FFFFFF;
		if (tagDataSkip >= tagDataSize) {
			tagDataSkip -= tagDataSize;
		}
		else {
			break;
		}
		tagIndex++;
		if (tagIndex >= mData.mAudioIndex.size()) {
			*lBytesRead = 0;
			*lSamplesRead = 0;
			return true;
		}
	}

	*lSamplesRead = min((tagDataSize - (uint32)tagDataSkip) / bytesPerSample, lCount);
	*lBytesRead = *lSamplesRead * bytesPerSample;

	if (lpBuffer) {
		if (cbBuffer < *lBytesRead)
			return false;

		_fseeki64(mData.mFile, tagInfo.mDataOffset + tagDataSkip, SEEK_SET);
		fread(lpBuffer, 1, *lBytesRead, mData.mFile);
	}

	return true;
}

const void *AudioSourceFLV::GetDirectFormat() {
	return &mData.mAudioFormat;
}

int AudioSourceFLV::GetDirectFormatLen() {
	return sizeof(mData.mAudioFormat.wfx) + mData.mAudioFormat.wfx.cbSize;
}

IVDXStreamSource::ErrorMode AudioSourceFLV::GetDecodeErrorMode() {
	return IVDXStreamSource::kErrorModeReportAll;
}

void AudioSourceFLV::SetDecodeErrorMode(IVDXStreamSource::ErrorMode mode) {
}

bool AudioSourceFLV::IsDecodeErrorModeSupported(IVDXStreamSource::ErrorMode mode) {
	return mode == IVDXStreamSource::kErrorModeReportAll;
}

bool AudioSourceFLV::IsVBR() {
	return (mData.mAudioCodec == kAC_MP3);
}

sint64 AudioSourceFLV::TimeToPositionVBR(sint64 us) {
	sint64 pOff, nOff;
	pOff = 0;
	nOff = 0;
	for (int i = 1; i < mData.mAudioIndex.size(); i++) {
		sint64 nTS = mData.mAudioIndex[i].mTimeStamp * 1000;
		pOff = nOff;
		nOff += mData.mAudioIndex[i-1].mDataSize;
		if (nTS >= us) {
			sint64 pTS = mData.mAudioIndex[i-1].mTimeStamp * 1000;
			sint64 ret = pOff + (sint64)((((double)(us - pTS) / (nTS - pTS)) * (nOff - pOff)) + 0.5);
			//char dbgBuff[256];
			//sprintf(dbgBuff, "TimeToPositionVBR: %d = %d", (uint32)us, (uint32)ret);
			//OutputDebugString(dbgBuff);
			return ret;
		}
	}
	return mData.mAudioTotalBytes;
}

sint64 AudioSourceFLV::PositionToTimeVBR(sint64 samples) {
	sint64 pOff, nOff;
	pOff = 0;
	nOff = 0;
	for (int i = 1; i < mData.mAudioIndex.size(); i++) {
		pOff = nOff;
		nOff += mData.mAudioIndex[i-1].mDataSize;
		if (nOff >= samples) {
			sint64 pTS = mData.mAudioIndex[i-1].mTimeStamp * 1000;
			sint64 nTS = mData.mAudioIndex[i].mTimeStamp * 1000;
			sint64 ret =  pTS + (sint64)((((double)(samples - pOff) / (nOff - pOff)) * (nTS - pTS)) + 0.5);
			//char dbgBuff[256];
			//sprintf(dbgBuff, "PositionToTimeVBR: %d = %d", (uint32)samples, (uint32)ret);
			//OutputDebugString(dbgBuff);
			return ret;
		}
	}
	return mData.mAudioEstimatedDuration;
}

void AudioSourceFLV::GetAudioSourceInfo(VDXAudioSourceInfo& info) {
	info.mFlags = 0;
}

///////////////////////////////////////////////////////////////////////////////

class InputFileFLV : public vdxunknown<IVDXInputFile> {
public:
	InputFileFLV(const VDInputDriverContext& context);

	void VDXAPIENTRY Init(const wchar_t *szFile, IVDXInputOptions *opts);
	bool VDXAPIENTRY Append(const wchar_t *szFile);

	bool VDXAPIENTRY PromptForOptions(VDXHWND, IVDXInputOptions **);
	bool VDXAPIENTRY CreateOptions(const void *buf, uint32 len, IVDXInputOptions **);
	void VDXAPIENTRY DisplayInfo(VDXHWND hwndParent);

	bool VDXAPIENTRY GetVideoSource(int index, IVDXVideoSource **);
	bool VDXAPIENTRY GetAudioSource(int index, IVDXAudioSource **);

protected:
	FLVData	mData;
	const VDInputDriverContext& mContext;
};

InputFileFLV::InputFileFLV(const VDInputDriverContext& context)
	: mContext(context)
{
}

void InputFileFLV::Init(const wchar_t *szFile, IVDXInputOptions *opts) {
	char abuf[1024];
	sint64 fileSize, filePos, fileRemain;
	uint8 buff[32];
	uint8* pBuff;
	uint8 flags;
	uint32 dataOffset;
	sint32 width, height;
	uint32 sampleRate, bitsPerSample, channelCount;
	uint32 firstVideoTS, firstAudioTS;

	wcstombs(abuf, szFile, 1024);
	abuf[1023] = 0;

	mData.mFile = fopen(abuf, "rb");
	if (!mData.mFile) {
		mContext.mpCallbacks->SetError("Unable to open file: %ls", szFile);
		return;
	}

	_fseeki64(mData.mFile, 0, SEEK_END);
	fileSize = _ftelli64(mData.mFile);
	_fseeki64(mData.mFile, 0, SEEK_SET);

	mData.mVideoCodec = kVC_None;
	firstVideoTS = 0xFFFFFFFF;
	width = 0;
	height = 0;
	mData.mAudioCodec = kAC_None;
	firstAudioTS = 0xFFFFFFFF;
	sampleRate = 0;
	bitsPerSample = 0;
	channelCount = 0;

	// Read the main header
	fread(buff, 1, 9, mData.mFile);
	pBuff = buff;
	if (GetInt32(pBuff) != 0x464C5601) { // 'F' 'L' 'V' 0x01
		mContext.mpCallbacks->SetError("This doesn't appear to be a FLV file.");
		return;
	}
	flags = GetInt8(pBuff);
	dataOffset = GetInt32(pBuff);

	// The remainder of the file is a series of 'tags', each of which holds
	// a video frame or chunk of audio
	filePos = dataOffset + 4;
	_fseeki64(mData.mFile, filePos, SEEK_SET);
	while ((fileRemain = fileSize - filePos) >= 11) {
		uint32 tagType, dataSize, timeStamp, mediaInfo;
		sint32 buffRemain;
		FLVTagInfo tagInfo;

		// Read tag header
		buffRemain = fread(buff, 1, (size_t)min(fileRemain, 32), mData.mFile);
		pBuff = buff;
		tagType = GetInt8(pBuff);
		dataSize = GetInt24(pBuff);
		timeStamp = GetInt24(pBuff);
		timeStamp |= GetInt8(pBuff) << 24;
		GetInt24(pBuff);

		if (((tagType == 0x09) || (tagType == 0x08)) && (dataSize != 0)) {
			// This byte is counted in dataSize but isn't part of the video frame
			// or audio data
			mediaInfo = GetInt8(pBuff);

			tagInfo.mDataOffset = filePos + 12;
			tagInfo.mDataSize = dataSize - 1;
			tagInfo.mTimeStamp = timeStamp;

			buffRemain = min(buffRemain - 12, (sint32)tagInfo.mDataSize);
			if ((fileRemain - 12) < tagInfo.mDataSize) {
				// Incomplete file, stop
				break;
			}

			if (tagType == 0x09) {
				// Video tag

				// Make sure the timestamps start at zero
				if (firstVideoTS == 0xFFFFFFFF) {
					firstVideoTS = tagInfo.mTimeStamp;
				}
				tagInfo.mTimeStamp -= firstVideoTS;

				if (mData.mVideoCodec == kVC_None) {
					switch (mediaInfo & 0x0F) {
						case 2: mData.mVideoCodec = kVC_H263;	 break;
						case 4: mData.mVideoCodec = kVC_VP6;	 break;
						case 5: mData.mVideoCodec = kVC_VP6A;	 break;
						case 3: mData.mVideoCodec = kVC_Screen;	 break;
						case 6: mData.mVideoCodec = kVC_Screen2; break;
					}
				}

				// Mark keyframes
				if ((mediaInfo >> 4) == 1) {
					tagInfo.mDataSize |= 0x01000000;
				}

				// There is some extra data at the beginning of VP6 frames in FLV
				// that the decoder doesn't want
				if (mData.mVideoCodec == kVC_VP6) {
					// Skip 1 byte
					tagInfo.mDataOffset += 1;
					tagInfo.mDataSize = max((sint32)tagInfo.mDataSize - 1, 0);
					GetInt8(pBuff);
					buffRemain -= 1;
				}
				else if (mData.mVideoCodec == kVC_VP6A) {
					// Skip 4 bytes
					tagInfo.mDataOffset += 4;
					tagInfo.mDataSize = max((sint32)tagInfo.mDataSize - 4, 0);
					GetInt32(pBuff);
					buffRemain -= 4;
				}

				// FLV container doesn't hold width/height, parse frame to find it
				if ((width == 0) || (height == 0)) {
					GetFrameSize(pBuff, buffRemain, mData.mVideoCodec, width, height);
				}

				mData.mVideoIndex.push_back(tagInfo);
			}
			else if (tagType == 0x08) {
				// Audio tag

				// Make sure the timestamps start at zero
				if (firstAudioTS == 0xFFFFFFFF) {
					firstAudioTS = tagInfo.mTimeStamp;
				}
				tagInfo.mTimeStamp -= firstAudioTS;

				if (mData.mAudioCodec == kAC_None) {
					switch (mediaInfo >> 4) {
						case 0: mData.mAudioCodec = kAC_PCM;		  break;
						case 1: mData.mAudioCodec = kAC_ADPCM;		  break;
						case 2: mData.mAudioCodec = kAC_MP3;		  break;
						case 5: mData.mAudioCodec = kAC_Nellymoser8M; break;
						case 6: mData.mAudioCodec = kAC_Nellymoser;	  break;
					}
					switch ((mediaInfo >> 2) & 0x03) {
						case 0: sampleRate =  5512; break;
						case 1: sampleRate = 11025; break;
						case 2: sampleRate = 22050; break;
						case 3: sampleRate = 44100; break;
					}
					bitsPerSample = ((mediaInfo >> 1) & 0x01) ? 16 : 8;
					channelCount = (mediaInfo & 0x01) ? 2 : 1;
				}

				mData.mAudioIndex.push_back(tagInfo);
			}
		}

		filePos = filePos + 11 + dataSize + 4;
		_fseeki64(mData.mFile, filePos, SEEK_SET);
	}

	BITMAPINFOHEADER vf;
	vf.biSize = sizeof(vf);
	vf.biWidth = width;
	vf.biHeight = height;
	vf.biPlanes = 1;
	vf.biBitCount = 24;
	if (mData.mVideoCodec == kVC_H263) {
		vf.biCompression = kFCC_FLV1;
	}
	else if ((mData.mVideoCodec == kVC_VP6) || (mData.mVideoCodec == kVC_VP6A)) {
		vf.biCompression = kFCC_VP6F;
	}
	else {
		vf.biCompression = 0;
	}
	vf.biSizeImage = (width * 4) * height;
	vf.biXPelsPerMeter = 0;
	vf.biYPelsPerMeter = 0;
	vf.biClrUsed = 0;
	vf.biClrImportant = 0;

	mData.mAudioTotalBytes = TotalStreamBytes(mData.mAudioIndex);
	MPEGLAYER3WAVEFORMAT af;
	if (mData.mAudioCodec == kAC_MP3) {
		int tagCount = mData.mAudioIndex.size();
		if (tagCount >= 2) {
			uint32 lastTS0 = mData.mAudioIndex[tagCount - 1].mTimeStamp;
			uint32 lastTS1 = mData.mAudioIndex[tagCount - 2].mTimeStamp;
			mData.mAudioEstimatedDuration = lastTS0 + lastTS0 - lastTS1;
		}
		else {
			mData.mAudioEstimatedDuration = 25;
		}
		af.wfx.wFormatTag = WAVE_FORMAT_MPEGLAYER3;
		af.wfx.nChannels = channelCount;
		af.wfx.nSamplesPerSec = sampleRate;
		af.wfx.nAvgBytesPerSec = (DWORD)((((double)mData.mAudioTotalBytes * 1000.0) / mData.mAudioEstimatedDuration) + 0.5);
		af.wfx.nBlockAlign = 1;
		af.wfx.wBitsPerSample = 0;
		af.wfx.cbSize = MPEGLAYER3_WFX_EXTRA_BYTES;
		af.wID = MPEGLAYER3_ID_MPEG;
		af.fdwFlags = MPEGLAYER3_FLAG_PADDING_ISO;
		af.nBlockSize = (WORD)(((double)mData.mAudioTotalBytes / (((double)sampleRate * mData.mAudioEstimatedDuration) / (1000.0 * 1152.0))) + 0.5);
		af.nFramesPerBlock = 1;
		af.nCodecDelay = 0;
	}
	else if (mData.mAudioCodec == kAC_PCM) {
		af.wfx.wFormatTag = WAVE_FORMAT_PCM;
		af.wfx.nChannels = channelCount;
		af.wfx.nSamplesPerSec = sampleRate;
		af.wfx.nAvgBytesPerSec = channelCount * sampleRate * (bitsPerSample / 8);
		af.wfx.nBlockAlign = channelCount * (bitsPerSample / 8);
		af.wfx.wBitsPerSample = bitsPerSample;
		af.wfx.cbSize = 0;
	}

	mData.mVideoFormat = vf;
	mData.mAudioFormat = af;
}

bool InputFileFLV::Append(const wchar_t *szFile) {
	return false;
}

bool InputFileFLV::PromptForOptions(VDXHWND, IVDXInputOptions **ppOptions) {
	*ppOptions = NULL;
	return false;
}

bool InputFileFLV::CreateOptions(const void *buf, uint32 len, IVDXInputOptions **ppOptions) {
	*ppOptions = NULL;
	return false;
}

void InputFileFLV::DisplayInfo(VDXHWND hwndParent) {
}

bool InputFileFLV::GetVideoSource(int index, IVDXVideoSource **ppVS) {
	*ppVS = NULL;

	if (index)
		return false;

	if (mData.mVideoIndex.size() == 0)
		return false;

	if ((mData.mVideoCodec != kVC_H263) &&
		(mData.mVideoCodec != kVC_VP6) &&
		(mData.mVideoCodec != kVC_VP6A))
	{
		return false;
	}

	IVDXVideoSource *pVS = new VideoSourceFLV(mData);
	if (!pVS)
		return false;

	*ppVS = pVS;
	pVS->AddRef();
	return true;
}

bool InputFileFLV::GetAudioSource(int index, IVDXAudioSource **ppAS) {
	*ppAS = NULL;

	if (index)
		return false;

	if (mData.mAudioIndex.size() == 0)
		return false;

	if ((mData.mAudioCodec != kAC_MP3) &&
		(mData.mAudioCodec != kAC_PCM))
	{
		return false;
	}

	IVDXAudioSource *pAS = new AudioSourceFLV(mData);
	if (!pAS)
		return false;

	*ppAS = pAS;
	pAS->AddRef();
	return true;
}

///////////////////////////////////////////////////////////////////////////////

class InputFileDriverFLV : public vdxunknown<IVDXInputFileDriver> {
public:
	InputFileDriverFLV(const VDInputDriverContext& context);
	~InputFileDriverFLV();

	int VDXAPIENTRY AddRef();
	int VDXAPIENTRY Release();

	int		VDXAPIENTRY DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize);
	bool	VDXAPIENTRY CreateInputFile(uint32 flags, IVDXInputFile **ppFile);

protected:
	int		mRefCount;

	const VDInputDriverContext& mContext;
};

InputFileDriverFLV::InputFileDriverFLV(const VDInputDriverContext& context)
	: mRefCount(0)
	, mContext(context)
{
}

InputFileDriverFLV::~InputFileDriverFLV() {
}

int VDXAPIENTRY InputFileDriverFLV::AddRef() {
	return ++mRefCount;
}

int VDXAPIENTRY InputFileDriverFLV::Release() {
	int rv = --mRefCount;
	if (!rv)
		delete this;
	return rv;
}

int VDXAPIENTRY InputFileDriverFLV::DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize) {
	return -1;
}

bool VDXAPIENTRY InputFileDriverFLV::CreateInputFile(uint32 flags, IVDXInputFile **ppFile) {
	IVDXInputFile *p = new InputFileFLV(mContext);
	if (!p)
		return false;

	*ppFile = p;
	p->AddRef();
	return true;
}

bool VDXAPIENTRY flv_create(const VDInputDriverContext *pContext, IVDXInputFileDriver **ppDriver) {
	IVDXInputFileDriver *p = new InputFileDriverFLV(*pContext);
	if (!p)
		return false;
	*ppDriver = p;
	p->AddRef();
	return true;
}

const uint8 flv_sig[] = {
	'F' , 0xFF,
	'L' , 0xFF,
	'V' , 0xFF,
	0x01, 0xFF,
};

const VDInputDriverDefinition flv_input={
	sizeof(VDInputDriverDefinition),
	VDInputDriverDefinition::kFlagSupportsVideo,
	0,
	sizeof flv_sig,
	flv_sig,
	L"*.flv",
	L"Flash Video (*.flv)|*.flv",
	L"Flash Video input driver",
	flv_create
};

const VDPluginInfo flv_info={
	sizeof(VDPluginInfo),
	L"Flash Video input driver",
	L"Moitah",
	L"Loads Flash Video files.",
	0x00030000,
	kVDPluginType_Input,
	0,
	kVDPlugin_APIVersion,
	kVDPlugin_APIVersion,
	kVDPlugin_InputDriverAPIVersion,
	kVDPlugin_InputDriverAPIVersion,
	&flv_input
};

///////////////////////////////////////////////////////////////////////////

const VDPluginInfo *const kPlugins[]={
	&flv_info,
	NULL
};

extern "C" __declspec(dllexport) const VDPluginInfo *const * VDXAPIENTRY VDGetPluginInfo() {
	return kPlugins;
}
