// ****************************************************************************
// 
// MP4 Input Driver Plugin for VirtualDub
// Copyright (C) 2008  Skakov Pavel (skakov@rain.ifmo.ru)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// ****************************************************************************

#define _CRT_SECURE_NO_WARNINGS
#include "vd2/plugin/vdplugin.h"
#include "vd2/plugin/vdinputdriver.h"
#include "vd2/VDXFrame/Unknown.h"
#include <windows.h>
#include <vfw.h>

typedef unsigned int uint;
typedef uint64 u64;
typedef uint32 u32;
typedef uint16 u16;
typedef uint8  u8;
typedef int64 i64;
typedef int32 i32;
typedef int16 i16;
typedef int8  i8;

template <class T>__forceinline const T& klMin(const T &a, const T &b) {return a < b ? a : b;}
template <class T>__forceinline const T& klMax(const T &a, const T &b) {return a > b ? a : b;}
template <class T>__forceinline void klSwap(T &a, T &b) {const T t = a; a = b; b = t;}

#define Warning(string) MessageBoxA(NULL, string, "MP4 input driver", MB_ICONWARNING | MB_APPLMODAL)

///////////////////////////////////////////////////////////////////////////////

struct MP4Box
{
	u64 size;
	u32 type;
	u8 extended_type[0x10];
};

struct MP4Track
{
	enum
	{
		tread_mdhd = 1,
		tread_hdlr = 2,
		tread_stsd = 4,
		tread_stts = 8,
		tread_ctts = 0x8000,
		tread_stsz = 0x10,
		tread_stsc = 0x20,
		tread_stco = 0x40,
		tread_stss = 0x4000,
		tread_stbl = 0x80, // sample_offset is built based on stsd, stts, stsz, stsc, stco
		tread_mdia = 0x100,
		tread_elst = 0x200,
	};
	wchar_t file_name[MAX_PATH];
	u16 tread;
	u32 timescale;
	u64 duration;
	u64 start_offset;
	u32 handler_type;
	char language[4];
	u16 data_reference_index;
	u32 sample_number;
	i64 *sample_time;
	i32 *sample_composite_time; // standard defines it as u32, but Nero AVC Muxer sometimes writes here negative i32 numbers
	u64 *sample_offset; // msb is non-key flag
	u32 *sample_size;
	u8  *sample_type;
	u32 *frame_to_sample;

	union
	{
		BITMAPINFOHEADER *video;
		VDXWAVEFORMATEX *audio;
	} format;
	u32 format_size;

	u64 total_size;
	u32 max_sample_size;
	bool cbr;

	MP4Track()
	{
		tread = 0;
		start_offset = 0;
		sample_time = NULL;
		sample_composite_time = NULL;
		sample_offset = NULL;
		sample_size = NULL;
		sample_type = NULL;
		frame_to_sample = NULL;
		format.video = NULL;
		cbr = false;
	}
	~MP4Track()
	{
		delete[] sample_time;
		delete[] sample_composite_time;
		delete[] sample_offset;
		delete[] sample_size;
		delete[] sample_type;
		delete[] frame_to_sample;
		delete[] (u8*)format.video;
	}
};

struct sort_time_table
{
	i64 time;
	u32 index;
};

///////////////////////////////////////////////////////////////////////////////

i64 MulDivLongLong(i64 x, u32 ml, u32 dv)
{
	u64 y = x >= 0 ? x : -x;
	u64 l = (u64)(u32)y * ml;
	u64 h = (y >> 0x20) * ml;
	h += l >> 0x20;
	l = (h % dv) << 0x20 | (u32)l;
	return h/dv << 0x20 | l/dv;
}

int __cdecl sort_time_compare(const void *a, const void *b)
{
	return ((sort_time_table*)a)->time != ((sort_time_table*)b)->time ? ((sort_time_table*)a)->time < ((sort_time_table*)b)->time ? -1 : 1 : ((sort_time_table*)a)->index - ((sort_time_table*)b)->index;
}

///////////////////////////////////////////////////////////////////////////////

class VDInputFilePluginMP4 : public vdxunknown<IVDXInputFile> {
public:
	VDInputFilePluginMP4(const VDXInputDriverContext& context);
	~VDInputFilePluginMP4();

	void VDXAPIENTRY Init(const wchar_t *szFile, IVDXInputOptions *opts);
	bool VDXAPIENTRY Append(const wchar_t *szFile);

	bool VDXAPIENTRY PromptForOptions(VDXHWND, IVDXInputOptions **);
	bool VDXAPIENTRY CreateOptions(const void *buf, uint32 len, IVDXInputOptions **);
	void VDXAPIENTRY DisplayInfo(VDXHWND hwndParent);

	bool VDXAPIENTRY GetVideoSource(int index, IVDXVideoSource **);
	bool VDXAPIENTRY GetAudioSource(int index, IVDXAudioSource **);

protected:
	u64  GetSize(void) {u64 result; *(u32*)&result = GetFileSize(m_hFile, (LPDWORD)&result + 1); return (u32)result != INVALID_FILE_SIZE || GetLastError() == NO_ERROR ? result : 0;}
	bool Seek(u64 offset) {return SetFilePointer(m_hFile, (u32)offset, (PLONG)&offset + 1, FILE_BEGIN) != INVALID_SET_FILE_POINTER || GetLastError() == NO_ERROR;}
	bool Read(void *pBuffer, u32 len) {DWORD count; return ReadFile(m_hFile, pBuffer, len, &count, NULL) ? count == len : false;}
	// reads 2-8 are endian-dependent, current implementation is little-endian
	u8   Read1() {u8  result; return Read(&result, 1) ? result : 0;}
	u16  Read2() {u16 result; return Read(&result, 2) ? _byteswap_ushort(result) : 0;}
	u32  Read3() {u32 result = 0; return Read((u8*)&result + 1, 3) ? _byteswap_ulong(result) : 0;}
	u32  Read4() {u32 result; return Read(&result, 4) ? _byteswap_ulong(result) : 0;}
	u64  Read8() {u64 result; return Read(&result, 8) ? _byteswap_uint64(result) : 0;}
	uint ReadV(u32 *pValue);
	uint ReadBox(MP4Box *pBox, u64 offset); // returns data offset in the box

	void ParseMoov(u64 pos, const u64 moov_end);
	void ParseMdia(u64 pos, const u64 mdia_end, MP4Track *pTrack);
	void ParseStbl(u64 pos, const u64 stbl_end, MP4Track *pTrack);
	uint ParseEsds(u64 pos, const u64 esds_end, u32 *bitrate_avg, u32 *codec_data_len, u64 *codec_data_pos);

	bool ParseAudioType(u64 pos, const u64 pos_end, u32 codingname, MP4Track *pTrack);
	bool ParseVideoType(u64 pos, const u64 pos_end, u32 codingname, MP4Track *pTrack);
	bool FillVideoMediaType(MP4Track *pTrack, u64 pos, u32 codingname, u32 codec_data_len, u32 pixel_aspect_x, u32 pixel_aspect_y);

	const VDXInputDriverContext& mContext;
	const wchar_t *mBaseFileName; // valid only during Init
	HANDLE m_hFile;
	MP4Track **m_VideoStreams, **m_AudioStreams;
	int m_nVideoStreams, m_nAudioStreams;
	uint32 mMajorBrand;
};

///////////////////////////////////////////////////////////////////////////////

class VDVideoDecoderModelPluginMP4 : public vdxunknown<IVDXVideoDecoderModel> {
public:
	VDVideoDecoderModelPluginMP4(const MP4Track *track);
	~VDVideoDecoderModelPluginMP4();

	void	VDXAPIENTRY Reset();
	void	VDXAPIENTRY SetDesiredFrame(sint64 frame_num);
	sint64	VDXAPIENTRY GetNextRequiredSample(bool& is_preroll);
	int		VDXAPIENTRY GetRequiredCount();

protected:
	const	MP4Track *mpTrack;

	u32 mDesired;
	u32 mPrevIP;
	u32 mLast;
	u32 mLastIP;
	u32 mCurrentI;

	sint64	mLastPrevFrame;
	sint64	mLastNextFrame;
	sint64	mLastBidirFrame;
	sint64	mTargetFrame;
	sint64	mNextFrame;
};

VDVideoDecoderModelPluginMP4::VDVideoDecoderModelPluginMP4(const MP4Track *track)
	: mpTrack(track)
{
}

VDVideoDecoderModelPluginMP4::~VDVideoDecoderModelPluginMP4() {
}

void VDVideoDecoderModelPluginMP4::Reset() {
	mLastPrevFrame = -1;
	mLastNextFrame = -1;
	mLastBidirFrame = -1;
}

void VDVideoDecoderModelPluginMP4::SetDesiredFrame(sint64 sample_num) {
	mDesired = (u32)sample_num;

	if (mpTrack->sample_type[mDesired] == kVDXVFT_Independent)
	{
		mCurrentI = mDesired;
		mPrevIP = mDesired;
		mLastIP = -1;
		mLast = -1;
		return;
	}
	if (mDesired == mLastIP || mDesired == mLast + 1)
		return;

	u32 PrevIP;
	for (PrevIP = mDesired ? mDesired - 1 : 0; PrevIP && mpTrack->sample_type[PrevIP] == kVDXVFT_Bidirectional; PrevIP--);

	u32 CurrentI;
	for (CurrentI = PrevIP; CurrentI && mpTrack->sample_type[CurrentI] != kVDXVFT_Independent; CurrentI--);

	mPrevIP = mpTrack->sample_type[mDesired] == kVDXVFT_Predicted ? mDesired : PrevIP;

	if (mLast < mDesired && mCurrentI == CurrentI)
		return;

	mCurrentI = CurrentI;
	mLastIP = -1;
	mLast = -1;
}

sint64 VDVideoDecoderModelPluginMP4::GetNextRequiredSample(bool& is_preroll) {
	if (mDesired == mLast || mDesired == mLastIP && (mLast + 1 == mpTrack->sample_number || mpTrack->sample_type[mLast + 1] != kVDXVFT_Bidirectional))
	{
		is_preroll = false;
		return -1;
	}

	if (mLast == -1)
	{
		mLastIP = mCurrentI;
		mLast = mCurrentI;
		is_preroll = mLastIP < mPrevIP;
		return mLast;
	}

	mLast++;
	if (mpTrack->sample_type[mLast] == kVDXVFT_Bidirectional)
		is_preroll = mLast < mDesired;
	else
	{
		mLastIP = mLast;
		is_preroll = mLastIP < mPrevIP;
		if (mpTrack->sample_type[mLast] == kVDXVFT_Independent)
			mCurrentI = mLast;
	}

	return mLast;
}

int VDVideoDecoderModelPluginMP4::GetRequiredCount() {
	return mNextFrame == -1 ? 0 : 1;
}

///////////////////////////////////////////////////////////////////////////////

class VDStreamSourcePluginMP4 : public vdxunknown<IVDXStreamSource> {
public:
	VDStreamSourcePluginMP4(const VDXInputDriverContext& context, VDInputFilePluginMP4 *parent, MP4Track *track);
	~VDStreamSourcePluginMP4();

	void		VDXAPIENTRY GetStreamSourceInfo(VDXStreamSourceInfo&);
	bool		VDXAPIENTRY Read(sint64 lStart, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead);

	const void *VDXAPIENTRY GetDirectFormat();
	int			VDXAPIENTRY GetDirectFormatLen();

	ErrorMode	VDXAPIENTRY GetDecodeErrorMode();
	void		VDXAPIENTRY SetDecodeErrorMode(ErrorMode mode);
	bool		VDXAPIENTRY IsDecodeErrorModeSupported(ErrorMode mode);

	bool		VDXAPIENTRY IsVBR();
	sint64		VDXAPIENTRY TimeToPositionVBR(sint64 us);
	sint64		VDXAPIENTRY PositionToTimeVBR(sint64 samples);

protected:
	bool Seek(u64 offset) {return SetFilePointer(mhFile, (u32)offset, (PLONG)&offset + 1, FILE_BEGIN) != INVALID_SET_FILE_POINTER || GetLastError() == NO_ERROR;}

	const VDXInputDriverContext& mContext;
	VDInputFilePluginMP4 *mParent;
	MP4Track *mpTrack;
	HANDLE mhFile;
};

VDStreamSourcePluginMP4::VDStreamSourcePluginMP4(const VDXInputDriverContext& context, VDInputFilePluginMP4 *parent, MP4Track *track)
	: mContext(context)
	, mParent(parent)
	, mpTrack(track)
{
	mParent->AddRef();
	mhFile = CreateFileW(mpTrack->file_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
}

VDStreamSourcePluginMP4::~VDStreamSourcePluginMP4() {
	CloseHandle(mhFile);
	mParent->Release();
}

void VDXAPIENTRY VDStreamSourcePluginMP4::GetStreamSourceInfo(VDXStreamSourceInfo& srcInfo) {
	u64 a = (u64)mpTrack->sample_number*mpTrack->timescale;
	u64 b = mpTrack->duration;
	while (a > (uint32)-1 || b > (uint32)-1)
	{
		a >>= 1;
		b >>= 1;
	}
	if (!a)
		a = 1;
	if (!b)
		b = 1;
	srcInfo.mSampleRate.mNumerator		= (u32)a;
	srcInfo.mSampleRate.mDenominator	= (u32)b;
	srcInfo.mSampleCount				= mpTrack->sample_number;
}

bool VDStreamSourcePluginMP4::Read(sint64 lStart64, uint32 lCount, void *lpBuffer, uint32 cbBuffer, uint32 *lBytesRead, uint32 *lSamplesRead) {
	*lBytesRead = 0;
	*lSamplesRead = 0;

	if (lStart64 >= mpTrack->sample_number)
		return true;
	if (lStart64 < 0)
		return false;

	u32 lStart = (u32)lStart64;

	if (lCount > mpTrack->sample_number - lStart)
		lCount = mpTrack->sample_number - lStart;

	if (!lpBuffer)
		for (; *lSamplesRead < lCount; (*lSamplesRead)++, lStart++)
			if (*lBytesRead + mpTrack->sample_size[lStart] >= *lBytesRead)
				*lBytesRead += mpTrack->sample_size[lStart];
			else
				break; // u32 overflow
	else
		if (cbBuffer < mpTrack->sample_size[lStart])
			return false;
		else
			for (DWORD read; *lSamplesRead < lCount && cbBuffer >= mpTrack->sample_size[lStart]; (*lSamplesRead)++, lStart++)
				if (Seek(mpTrack->sample_offset[lStart]) && 
					ReadFile(mhFile, lpBuffer, mpTrack->sample_size[lStart], &read, NULL) && mpTrack->sample_size[lStart] == read)
				{
					*lBytesRead += read;
					*(u8**)&lpBuffer += read;
					cbBuffer -= read;
				}
				else
				{
					mContext.mpCallbacks->SetError("Unable to read sample %i from stream '%4s'", lStart, &mpTrack->handler_type);
					break;
				}

	return true;
}

const void *VDStreamSourcePluginMP4::GetDirectFormat() {
	return mpTrack->format.video;
}

int VDStreamSourcePluginMP4::GetDirectFormatLen() {
	return mpTrack->format_size;
}

IVDXStreamSource::ErrorMode VDStreamSourcePluginMP4::GetDecodeErrorMode() {
	return IVDXStreamSource::kErrorModeReportAll;
}

void VDStreamSourcePluginMP4::SetDecodeErrorMode(IVDXStreamSource::ErrorMode mode) {
}

bool VDStreamSourcePluginMP4::IsDecodeErrorModeSupported(IVDXStreamSource::ErrorMode mode) {
	return mode == IVDXStreamSource::kErrorModeReportAll;
}

bool VDStreamSourcePluginMP4::IsVBR() {
	return true;
}

sint64 VDStreamSourcePluginMP4::TimeToPositionVBR(sint64 us) {
	us = MulDivLongLong(us, mpTrack->timescale, 1000000);
	if (us < 0 || (u64)us <= mpTrack->start_offset)
		return 0;
	us -= mpTrack->start_offset;
	u32 sample = 0;
	i64 delta = us;
	for (u32 i = 0; i < mpTrack->sample_number; i++)
	{
		i64 time = mpTrack->sample_time[i] + mpTrack->sample_composite_time[i] - mpTrack->sample_composite_time[0];
		if (us >= time && delta > us - time)
		{
			delta = us - time;
			sample = i;
		}
	}
	return sample;
}

sint64 VDStreamSourcePluginMP4::PositionToTimeVBR(sint64 sample) {
	return MulDivLongLong((sample >= mpTrack->sample_number ? mpTrack->duration : mpTrack->sample_time[sample] + mpTrack->sample_composite_time[sample]) + mpTrack->start_offset, 1000000, mpTrack->timescale);
}

///////////////////////////////////////////////////////////////////////////////

class VDVideoSourcePluginMP4 : public VDStreamSourcePluginMP4, public IVDXVideoSource {
public:
	VDVideoSourcePluginMP4(const VDXInputDriverContext& context, VDInputFilePluginMP4 *parent, MP4Track *track);
	~VDVideoSourcePluginMP4();

	int			VDXAPIENTRY AddRef();
	int			VDXAPIENTRY Release();
	void *		VDXAPIENTRY AsInterface(uint32 iid);

	void		VDXAPIENTRY GetVideoSourceInfo(VDXVideoSourceInfo& info);

	bool		VDXAPIENTRY CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel);
	bool		VDXAPIENTRY CreateVideoDecoder(IVDXVideoDecoder **ppDecoder);

	void		VDXAPIENTRY GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo);

	bool		VDXAPIENTRY IsKey(sint64 lSample);

	sint64		VDXAPIENTRY GetFrameNumberForSample(sint64 sample_num);
	sint64		VDXAPIENTRY GetSampleNumberForFrame(sint64 display_num);
	sint64		VDXAPIENTRY GetRealFrame(sint64 display_num);

	sint64		VDXAPIENTRY GetSampleBytePosition(sint64 sample_num);
};

VDVideoSourcePluginMP4::VDVideoSourcePluginMP4(const VDXInputDriverContext& context, VDInputFilePluginMP4 *parent, MP4Track *track)
	: VDStreamSourcePluginMP4(context, parent, track)
{
}

VDVideoSourcePluginMP4::~VDVideoSourcePluginMP4() {
}

int VDXAPIENTRY VDVideoSourcePluginMP4::AddRef() {
	return VDStreamSourcePluginMP4::AddRef();
}

int VDXAPIENTRY VDVideoSourcePluginMP4::Release() {
	return VDStreamSourcePluginMP4::Release();
}

void *VDXAPIENTRY VDVideoSourcePluginMP4::AsInterface(uint32 iid) {
	if (iid == IVDXVideoSource::kIID)
		return static_cast<IVDXVideoSource *>(this);

	return VDStreamSourcePluginMP4::AsInterface(iid);
}

void VDXAPIENTRY VDVideoSourcePluginMP4::GetVideoSourceInfo(VDXVideoSourceInfo& info) {
	const VDXBITMAPINFOHEADER *hdr = (const VDXBITMAPINFOHEADER *)GetDirectFormat();

	info.mFlags			= 0;
	info.mWidth			= mpTrack->format.video->biWidth;
	info.mHeight		= mpTrack->format.video->biHeight;
	info.mDecoderModel	= VDXVideoSourceInfo::kDecoderModelCustom;
//	info.mDecoderModel	= VDXVideoSourceInfo::kDecoderModelDefaultIP;
}

bool VDXAPIENTRY VDVideoSourcePluginMP4::CreateVideoDecoderModel(IVDXVideoDecoderModel **ppModel) {
	*ppModel = NULL;

	VDVideoDecoderModelPluginMP4 *pModel = new VDVideoDecoderModelPluginMP4(mpTrack);
	if (!pModel)
		return false;

	pModel->AddRef();

	*ppModel = pModel;
	return true;
}

bool VDXAPIENTRY VDVideoSourcePluginMP4::CreateVideoDecoder(IVDXVideoDecoder **ppDecoder) {
	*ppDecoder = NULL;
	return false;
}

void VDXAPIENTRY VDVideoSourcePluginMP4::GetSampleInfo(sint64 sample_num, VDXVideoFrameInfo& frameInfo) {
	frameInfo.mFrameType = mpTrack->sample_type[sample_num];
	frameInfo.mTypeChar = frameInfo.mFrameType == kVDXVFT_Independent ? 'I' : frameInfo.mFrameType == kVDXVFT_Predicted ? 'P' : 'B';
	frameInfo.mBytePosition = mpTrack->sample_offset[sample_num];
}

bool VDXAPIENTRY VDVideoSourcePluginMP4::IsKey(sint64 sample) {
	return mpTrack->sample_type[sample] == kVDXVFT_Independent;
}

sint64 VDXAPIENTRY VDVideoSourcePluginMP4::GetFrameNumberForSample(sint64 sample_num) {
	for (u32 i = 0; i < mpTrack->sample_number; i++)
		if (mpTrack->frame_to_sample[i] == (u32)sample_num)
		{
			sample_num = i;
			break;
		}
	return sample_num;
}

sint64 VDXAPIENTRY VDVideoSourcePluginMP4::GetSampleNumberForFrame(sint64 frame_num) {
//	return frame_num;
	return mpTrack->frame_to_sample[(u32)frame_num];
}

sint64 VDXAPIENTRY VDVideoSourcePluginMP4::GetRealFrame(sint64 frame_num) {
	return frame_num;
}

sint64 VDXAPIENTRY VDVideoSourcePluginMP4::GetSampleBytePosition(sint64 sample_num) {
	return mpTrack->sample_offset[sample_num];
}

///////////////////////////////////////////////////////////////////////////////

VDInputFilePluginMP4::VDInputFilePluginMP4(const VDXInputDriverContext& context)
	: mContext(context)
{
	m_hFile = INVALID_HANDLE_VALUE;
	m_VideoStreams = NULL;
	m_AudioStreams = NULL;
	m_nVideoStreams = 0;
	m_nAudioStreams = 0;
}

VDInputFilePluginMP4::~VDInputFilePluginMP4()
{
	if (m_hFile != INVALID_HANDLE_VALUE)
		CloseHandle(m_hFile);
	while (m_nVideoStreams)
		delete m_VideoStreams[--m_nVideoStreams];
	free(m_VideoStreams);
	while (m_nAudioStreams)
		delete m_AudioStreams[--m_nAudioStreams];
	free(m_AudioStreams);
}

bool VDInputFilePluginMP4::Append(const wchar_t *szFile) {
	return false;
}

bool VDInputFilePluginMP4::PromptForOptions(VDXHWND, IVDXInputOptions **ppOptions) {
	*ppOptions = NULL;
	return false;
}

bool VDInputFilePluginMP4::CreateOptions(const void *buf, uint32 len, IVDXInputOptions **ppOptions) {
	*ppOptions = NULL;
	return false;
}

void VDInputFilePluginMP4::DisplayInfo(VDXHWND hwndParent) {
}

bool VDInputFilePluginMP4::GetVideoSource(int index, IVDXVideoSource **ppVS) {
	*ppVS = NULL;

	if (index >= m_nVideoStreams)
		return false;

	VDVideoSourcePluginMP4 *pVS = new VDVideoSourcePluginMP4(mContext, this, m_VideoStreams[index]);
	if (!pVS)
		return false;

	pVS->AddRef();

	*ppVS = pVS;
	return true;
}

bool VDInputFilePluginMP4::GetAudioSource(int index, IVDXAudioSource **ppAS) {
	*ppAS = NULL;

	return false;
}

uint VDInputFilePluginMP4::ReadV(u32 *pValue)
{
	u32 result = 0;
	uint nBytes = 0;
	u8 x;
	do
	{
		x = Read1();
		result = (result << 7) | (x & 0x7F);
	} while (++nBytes < 4 && x & 0x80);
	if (pValue)
		*pValue = result;
	return nBytes;
}

uint VDInputFilePluginMP4::ReadBox(MP4Box *pBox, u64 offset)
{
	if (!Seek(offset))
	{
		ZeroMemory(pBox, sizeof(MP4Box));
		return 0;
	}
	pBox->size = Read4();
	pBox->type = Read4();
	uint len = 8;
	if (pBox->size == 1)
	{
		pBox->size = Read8();
		len += 8;
	}
	if (pBox->type == 'uuid')
	{
		if (!Read(pBox->extended_type, 0x10))
			ZeroMemory(pBox->extended_type, 0x10);
		len += 0x10;
	}
	return len;
}

void VDInputFilePluginMP4::Init(const wchar_t *szFile, IVDXInputOptions *opts) {
	m_hFile = CreateFileW(szFile, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

	if (m_hFile == INVALID_HANDLE_VALUE)
		mContext.mpCallbacks->SetError("Unable to open file");
	else
	{
		mBaseFileName = szFile;
		bool result = false;
		MP4Box box;
		u32 offset = ReadBox(&box, 0);
		if (box.size >= 0x10 && box.type == 'ftyp')
		{
			u32 type = mMajorBrand = Read4();
			for (u64 pos = box.size;;)
			{
				offset = ReadBox(&box, pos);
				if (box.type != 'moov')
					if ((i64)box.size >= offset)
					{
						pos += box.size;
						continue;
					}
					else
					{
						mContext.mpCallbacks->SetError("No 'moov' box");
						break;
					}
				ParseMoov(pos + offset, box.size ? pos + box.size : -1ll);
				if (!m_nVideoStreams && !m_nAudioStreams)
					mContext.mpCallbacks->SetError("No valid video or audio streams found");
				break;
			}
		}
		else
			mContext.mpCallbacks->SetError("Unknown or corrupt file");
	}
}

void VDInputFilePluginMP4::ParseMoov(u64 pos, const u64 moov_end)
{
	MP4Box box;
	u32 offset;
	while (pos < moov_end)
	{
		offset = ReadBox(&box, pos);
		if (box.type != 'trak')
			if ((i64)box.size >= offset)
			{
				pos += box.size;
				continue;
			}
			else
				break;
		u64 pos2 = pos + offset;
		pos = box.size ? pos + box.size : -1ll;
		MP4Track *pTrack = new MP4Track;
		while (pos2 < pos)
		{
			offset = ReadBox(&box, pos2);

			switch (box.type)
			{
				case 'mdia':
					if (!(pTrack->tread & MP4Track::tread_mdia))
						ParseMdia(pos2 + offset, box.size ? pos2 + box.size : -1ll, pTrack);
					break;
				case 'edts':
					if (!(pTrack->tread & MP4Track::tread_elst))
					{
						u64 pos3 = pos2 + offset;
						pos2 = box.size ? pos2 + box.size : -1ll;
						while (pos3 < pos2)
						{
							offset = ReadBox(&box, pos3);
							if (box.type != 'elst')
								if ((i64)box.size >= offset)
								{
									pos3 += box.size;
									continue;
								}
								else
									break;
							const u32 version = Read4();
							if (version & 0xFEFFFFFF)
							{
								pos3 += box.size;
								continue;
							}
							pTrack->tread |= MP4Track::tread_elst;
							if (!box.size)
								box.size = GetSize() - pos3;
							if (box.size >= offset + (version ? 0x30 : 0x20))
							{
								u32 edit_number = (u32)klMin((u64)Read4(), (box.size - offset - 8) / (version ? 0x14 : 0x0C));
								u32 i;
								u64 start_offset = 0;
								for (i = 0; i < edit_number; i++)
								{
									u64 segment_duration = version ? Read8() : Read4();
									i64 media_time = version ? Read8() : (i32)Read4();
									i16 media_rate_integer = Read2();
									i16 media_rate_fraction = Read2();
									if (media_time == -1)
										start_offset += segment_duration;
									else
									{
										if (i != edit_number - 1 || media_rate_integer != 1 || media_rate_fraction)
											Warning("'elst' box is too complex");
										break;
									}
								}
								pTrack->start_offset = start_offset;
							}
							break;
						}
						continue;
					}
			}
			if ((i64)box.size >= offset)
				pos2 += box.size;
			else
				break;
		}
		if (!(~pTrack->tread & (MP4Track::tread_mdhd | MP4Track::tread_hdlr | MP4Track::tread_stbl)))
			if (pTrack->handler_type == 'vide')
			{
				if (MP4Track **p = (MP4Track **)realloc(m_VideoStreams, sizeof(MP4Track *) * (m_nVideoStreams + 1)))
				{
					m_VideoStreams = p;
					m_VideoStreams[m_nVideoStreams] = pTrack;
					m_nVideoStreams++;
					pTrack = NULL;
				}
			}
			else if (pTrack->handler_type == 'soun')
			{
				if (MP4Track **p = (MP4Track **)realloc(m_AudioStreams, sizeof(MP4Track *) * (m_nAudioStreams + 1)))
				{
					m_AudioStreams = p;
					m_AudioStreams[m_nAudioStreams] = pTrack;
					m_nAudioStreams++;
					pTrack = NULL;
				}
			}
		if (pTrack && pTrack->tread & MP4Track::tread_hdlr && (pTrack->handler_type == 'vide'))// || pTrack->handler_type == 'soun'))
		{
			char buf[0x30];
			strcpy(buf, "Partially unrecognized stream skipped (    )");
			*(u32*)(buf + 39) = _byteswap_ulong(pTrack->handler_type);
			Warning(buf);
		}
		delete pTrack;
	}
}

void VDInputFilePluginMP4::ParseMdia(u64 pos, const u64 mdia_end, MP4Track *pTrack)
{
	u64 stbl_pos = 0, stbl_end = 0;
	pTrack->tread |= MP4Track::tread_mdia;

	while (pos < mdia_end)
	{
		MP4Box box;
		u32 offset = ReadBox(&box, pos);

		switch (box.type)
		{
			case 'mdhd':
				if (!(pTrack->tread & MP4Track::tread_mdhd))
				{
					const u32 version = Read4();
					if (!(version & 0xFEFFFFFF) && Seek(pos + offset + (version ? 0x14 : 0x0C)))
					{
						pTrack->timescale = Read4();
						Seek(pos + offset + (version ? 0x20 : 0x14));
						u16 language = Read2();
						pTrack->language[0] = (language >> 10 & 0x1F) + 0x60;
						pTrack->language[1] = (language >>  5 & 0x1F) + 0x60;
						pTrack->language[2] = (language       & 0x1F) + 0x60;
						pTrack->language[3] = 0;
						if (pTrack->timescale)
						{
							pTrack->tread |= MP4Track::tread_mdhd;
							if ((stbl_pos || stbl_end) && !(pTrack->tread & (MP4Track::tread_hdlr | MP4Track::tread_stbl) ^ MP4Track::tread_hdlr))
								ParseStbl(stbl_pos, stbl_end, pTrack);
						}
					}
				}
				break;
			case 'hdlr':
				if (!(pTrack->tread & MP4Track::tread_hdlr) && !(Read4() & 0xFF000000))
				{
					Read4(); // pre_defined = 0
					pTrack->handler_type = Read4();
					pTrack->tread |= MP4Track::tread_hdlr;
					if ((stbl_pos || stbl_end) && !(pTrack->tread & (MP4Track::tread_mdhd | MP4Track::tread_stbl) ^ MP4Track::tread_mdhd))
						ParseStbl(stbl_pos, stbl_end, pTrack);
				}
				break;
			case 'minf':
			{
				u64 pos2 = pos + offset;
				pos = box.size ? pos + box.size : -1ll;
				while (pos2 < pos)
				{
					offset = ReadBox(&box, pos2);
					// TODO: parse 'dref' box
					if (box.type != 'stbl')
						if ((i64)box.size >= offset)
						{
							pos2 += box.size;
							continue;
						}
						else
							break;
					if (!(pTrack->tread & (MP4Track::tread_mdhd | MP4Track::tread_hdlr | MP4Track::tread_stbl) ^ (MP4Track::tread_mdhd | MP4Track::tread_hdlr)))
						ParseStbl(pos2 + offset, box.size ? pos2 + box.size : -1ll, pTrack);
					else
					{
						stbl_pos = pos2 + offset;
						stbl_end = box.size ? pos2 + box.size : -1ll;
					}
					break;
				}
				continue;
			}
		}
		if ((i64)box.size >= offset)
			pos += box.size;
		else
			break;
	}
}

void VDInputFilePluginMP4::ParseStbl(u64 pos, const u64 stbl_end, MP4Track *pTrack)
{
	struct CHUNK_INFO
	{
		u32 first_chunk, samples_per_chunk, sample_description_index;
	} *chunk_info = NULL;
	u64 *chunk_offset = NULL;
	u32 *key_sample = NULL;
	u32 composite_time_number = 0;
	u32 key_sample_number;
	u32 chunk_info_number;
	u32 chunk_number;
	u32 sample_number_stsz;
	u32 active_sample_description_index;

	while (pos < stbl_end)
	{
		MP4Box box;
		u32 offset = ReadBox(&box, pos);

		switch (box.type)
		{
			case 'stsd':
				if (!(pTrack->tread & MP4Track::tread_stsd) && !(Read4() & 0xFF000000))
				{
					u32 entry_count = Read4();
					u64 pos2 = pos + offset + 8;
					pos = box.size ? pos + box.size : -1ll;
					for (u32 i = 0; !(pTrack->tread & MP4Track::tread_stsd) && pos2 < pos && i < entry_count; i++)
					{
						offset = ReadBox(&box, pos2);
						Seek(pos2 + offset + 6);
						pTrack->data_reference_index = Read2();
						if (
							(pTrack->handler_type == 'vide' && ParseVideoType(pos2 + offset, box.size ? pos2 + box.size : -1ll, box.type, pTrack))
//							(pTrack->handler_type == 'soun' && ParseAudioType(pos2 + offset, box.size ? pos2 + box.size : -1ll, box.type, pTrack))
							)
						{
							pTrack->tread |= MP4Track::tread_stsd;
							active_sample_description_index = i + 1;
						}
						if ((i64)box.size >= offset)
						{
							pos2 += box.size;
							continue;
						}
						else
							break;
					}
				}
				continue;
			case 'stts':
				if (!(pTrack->tread & MP4Track::tread_stts) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 0x10)
					{
						u32 entry_count = (u32)klMin((u64)Read4(), (box.size - offset - 8) / 8);
						u32 sample_number = 0;
						struct stts_entry
						{
							u32 sample_count, sample_delta;
						};
						if (stts_entry *table = new stts_entry[entry_count])
						{
							for (u32 i = 0; i < entry_count; i++)
							{
								table[i].sample_count = Read4();
								table[i].sample_delta = Read4();
								if (sample_number + table[i].sample_count < sample_number) // u32 overflow
									entry_count = i;
								else
									sample_number += table[i].sample_count;
							}
							if (i64 *time = new i64[sample_number + 1])
							{
								u32 sample_index = 0;
								time[0] = 0;
								for (u32 i = 0; i < entry_count; i++)
									for (; table[i].sample_count--; sample_index++)
										time[sample_index + 1] = time[sample_index] + table[i].sample_delta;
								pTrack->sample_number = sample_number;
								pTrack->sample_time = time;
								pTrack->tread |= MP4Track::tread_stts;
							}
							delete[] table;
						}
					}
				}
				break;
			case 'ctts':
				if (!(pTrack->tread & MP4Track::tread_ctts) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 0x10)
					{
						u32 entry_count = (u32)klMin((u64)Read4(), (box.size - offset - 8) / 8);
						struct ctts_entry
						{
							u32 sample_count, sample_offset;
						};
						if (ctts_entry *table = new ctts_entry[entry_count])
						{
							for (u32 i = 0; i < entry_count; i++)
							{
								table[i].sample_count = Read4();
								table[i].sample_offset = Read4();
								if (composite_time_number + table[i].sample_count < composite_time_number) // u32 overflow
									entry_count = i;
								else
									composite_time_number += table[i].sample_count;
							}
							if (composite_time_number && (pTrack->sample_composite_time = new i32[composite_time_number]))
							{
								u32 sample_index = 0;
								for (u32 i = 0; i < entry_count; i++)
									for (; table[i].sample_count--; sample_index++)
										pTrack->sample_composite_time[sample_index] = table[i].sample_offset;
								pTrack->tread |= MP4Track::tread_ctts;
							}
							delete[] table;
						}
					}
				}
				break;
			case 'cttx':
				if (!(pTrack->tread & MP4Track::tread_ctts) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 0x15)
					{
						u32 entry_size = Read4() & 0xFF;
						u32 table_size = Read4();
						if (!table_size)
							table_size = 1u << entry_size;
						if ((entry_size == 2 || entry_size == 4 || entry_size == 8) && table_size <= 1u << entry_size && box.size >= offset + 0x11 + table_size*4)
						{
							composite_time_number = (u32)klMin((u64)Read4(), (box.size - offset - 0x10 - table_size*4)*(8/entry_size));
							if (i32 *table = new i32[table_size])
							{
								for (u32 i = 0; i < table_size; i++)
									table[i] = Read4();
								if (composite_time_number && (pTrack->sample_composite_time = new i32[composite_time_number]))
								{
									u8 index;
									if (entry_size == 2)
										for (u32 i = 0; i < composite_time_number; i++)
										{
											if (i & 3)
												index <<= 2;
											else
												index = Read1();
											pTrack->sample_composite_time[i] = (u32)(index >> 6) < table_size ? table[index >> 6] : 0;
										}
									else if (entry_size == 4)
										for (u32 i = 0; i < composite_time_number; i++)
										{
											if (i & 1)
												index <<= 4;
											else
												index = Read1();
											pTrack->sample_composite_time[i] = (u32)(index >> 4) < table_size ? table[index >> 4] : 0;
										}
									else // if (entry_size == 8)
										for (u32 i = 0; i < composite_time_number; i++)
										{
											index = Read1();
											pTrack->sample_composite_time[i] = index < table_size ? table[index] : 0;
										}
									pTrack->tread |= MP4Track::tread_ctts;
								}
								delete[] table;
							}
						}
					}
				}
				break;
			case 'stss':
				if (!key_sample && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 8)
					{
						key_sample_number = (u32)klMin((u64)Read4(), (box.size - offset - 8) / 4);
						if (!key_sample_number)
						{
							if (key_sample = new u32[key_sample_number = 1])
								key_sample[0] = 0;
						}
						else
							if (key_sample = new u32[key_sample_number])
							{
								for (u32 i = 0; i < key_sample_number; i++)
									key_sample[i] = Read4() - 1;
							}
					}
				}
				break;
			case 'stsz':
				if (!(pTrack->tread & MP4Track::tread_stsz) && !(Read4() & 0xFF000000))
				{
					u32 sample_size = Read4();
					sample_number_stsz = Read4();
					if (sample_size)
					{
						if (pTrack->sample_size = new u32[sample_number_stsz])
						{
							for (u32 i = 0; i < sample_number_stsz; i++)
								pTrack->sample_size[i] = sample_size;
							pTrack->total_size = (u64)sample_size * sample_number_stsz;
							pTrack->max_sample_size = sample_size;
							pTrack->cbr = true;
							pTrack->tread |= MP4Track::tread_stsz;
						}
					}
					else
					{
						if (!box.size)
							box.size = GetSize() - pos;
						if (box.size >= offset + 0x10)
						{
							if (sample_number_stsz > (box.size - offset - 0x0C) / 4)
								sample_number_stsz = (u32)((box.size - offset - 0x0C) / 4);
							if (pTrack->sample_size = new u32[sample_number_stsz])
							{
								pTrack->total_size = 0;
								pTrack->max_sample_size = 0;
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									pTrack->sample_size[i] = Read4();
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
								pTrack->tread |= MP4Track::tread_stsz;
							}
						}
					}
				}
				break;
			case 'stz2':
				if (!(pTrack->tread & MP4Track::tread_stsz) && !(Read4() & 0xFF000000))
				{
					u8 field_size = Read4() & 0xFF;
					if (!box.size)
						box.size = GetSize() - pos;
					if ((field_size == 4 || field_size == 8 || field_size == 16 || field_size == 12 || field_size == 24) && box.size >= offset + 0x0C + (field_size + 7)/8)
					{
						sample_number_stsz = (u32)klMin((u64)Read4(), (box.size - offset - 0x0C) * 8 / field_size);
						if (pTrack->sample_size = new u32[sample_number_stsz])
						{
							pTrack->total_size = 0;
							pTrack->max_sample_size = 0;
							if (field_size == 4)
							{
								u8 index;
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									if (i & 1)
										index <<= 4;
									else
										index = Read1();
									pTrack->sample_size[i] = index >> 4;
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
							}
							else if (field_size == 8)
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									pTrack->sample_size[i] = Read1();
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
							else if (field_size == 16)
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									pTrack->sample_size[i] = Read2();
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
							else if (field_size == 12)
							{
								u32 index;
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									if (i & 1)
										index = index << 12 & 0xFFF000;
									else
										index = Read3();
									pTrack->sample_size[i] = index >> 12;
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
							}
							else if (field_size == 24)
								for (u32 i = 0; i < sample_number_stsz; i++)
								{
									pTrack->sample_size[i] = Read3();
									pTrack->total_size += pTrack->sample_size[i];
									if (pTrack->max_sample_size < pTrack->sample_size[i])
										pTrack->max_sample_size = pTrack->sample_size[i];
								}
							pTrack->tread |= MP4Track::tread_stsz;
						}
					}
				}
				break;
			case 'stsc':
				if (!(pTrack->tread & MP4Track::tread_stsc) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 0x14)
					{
						chunk_info_number = (u32)klMin((u64)Read4(), (box.size - offset - 8) / 0x0C);
						if (chunk_info = new CHUNK_INFO[chunk_info_number])
						{
							for (u32 i = 0; i < chunk_info_number; i++)
							{
								chunk_info[i].first_chunk = Read4();
								chunk_info[i].samples_per_chunk = Read4();
								chunk_info[i].sample_description_index = Read4();
							}
							pTrack->tread |= MP4Track::tread_stsc;
						}
					}
				}
				break;
			case 'stsx':
				if (!(pTrack->tread & MP4Track::tread_stsc) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + 0x15)
					{
						u64 pos2 = pos + offset + 8;
						chunk_info_number = 0;
						for (u32 i = Read4(); i-- && pos2 < pos + box.size - 0x0C;)
						{
							Seek(pos2 + 4);
							u8 field_size = Read4() & 0xFF;
							if (field_size != 8 && field_size != 16)
								break;
							u32 table_chunk_number = (u32)klMin((u64)Read4(), (pos - pos2 + box.size - 0x0C)/(field_size/8));
							if (chunk_info_number + table_chunk_number < chunk_info_number) // u32 overflow
								break;
							chunk_info_number += table_chunk_number;
							pos2 += (u64)table_chunk_number * (field_size/8) + 0x0C;
						}
						if (chunk_info_number && (chunk_info = new CHUNK_INFO[chunk_info_number]))
						{
							u32 chunk_info_index = 0;
							Seek(pos + offset + 8);
							while (chunk_info_index < chunk_info_number)
							{
								u32 sample_description_index = Read4();
								u8 field_size = Read4() & 0xFF;
								u32 table_chunk_number = klMin(Read4(), chunk_info_number - chunk_info_index);
								if (field_size == 8)
									while (table_chunk_number--)
									{
										chunk_info[chunk_info_index].first_chunk = chunk_info_index + 1;
										chunk_info[chunk_info_index].samples_per_chunk = Read1();
										chunk_info[chunk_info_index].sample_description_index = sample_description_index;
										chunk_info_index++;
									}
								else
									while (table_chunk_number--)
									{
										chunk_info[chunk_info_index].first_chunk = chunk_info_index + 1;
										chunk_info[chunk_info_index].samples_per_chunk = Read2();
										chunk_info[chunk_info_index].sample_description_index = sample_description_index;
										chunk_info_index++;
									}
							}
							pTrack->tread |= MP4Track::tread_stsc;
						}
					}
				}
				break;
			case 'stco':
			case 'co64':
				if (!(pTrack->tread & MP4Track::tread_stco) && !(Read4() & 0xFF000000))
				{
					if (!box.size)
						box.size = GetSize() - pos;
					if (box.size >= offset + (box.type == 'stco' ? 0x0C : 0x10))
					{
						chunk_number = (u32)klMin((u64)Read4(), (box.size - offset - 8) / (box.type == 'stco' ? 4 : 8));
						if (chunk_offset = new u64[chunk_number])
						{
							if (box.type == 'stco')
								for (u32 i = 0; i < chunk_number; i++)
									chunk_offset[i] = Read4();
							else
								for (u32 i = 0; i < chunk_number; i++)
									chunk_offset[i] = Read8();
							pTrack->tread |= MP4Track::tread_stco;
						}
					}
				}
				break;
		}
		if ((i64)box.size >= offset)
			pos += box.size;
		else
			break;
	}
	if (pTrack->tread & MP4Track::tread_stts)
	{
		if (pTrack->tread & MP4Track::tread_stsz)
		{
			if (pTrack->sample_number > sample_number_stsz)
				pTrack->sample_number = sample_number_stsz;
			if (!(~pTrack->tread & (MP4Track::tread_stsd | MP4Track::tread_stsc | MP4Track::tread_stco)) && pTrack->sample_number)
				if (pTrack->sample_type = new u8[pTrack->sample_number])
					if (pTrack->sample_offset = new u64[pTrack->sample_number])
					{
						u64 current_offset;
						u32 chunk_index = 0;
						u32 current_chunk_info_index = 0;
						u32 current_chunk_samples_number = 0;
						bool sample_valid;
						u32 sample_index;
						for (sample_index = 0; sample_index < pTrack->sample_number;)
						{
							if (!current_chunk_samples_number)
							{
								if (chunk_index >= chunk_number)
									break;
								while (current_chunk_info_index < chunk_info_number - 1 && chunk_info[current_chunk_info_index + 1].first_chunk - 1 <= chunk_index)
									current_chunk_info_index++;
								current_chunk_samples_number = chunk_info[current_chunk_info_index].samples_per_chunk;
								current_offset = chunk_offset[chunk_index];
								sample_valid = chunk_info[current_chunk_info_index].sample_description_index == active_sample_description_index;
								chunk_index++;
								continue;
							}
							pTrack->sample_offset[sample_index] = sample_valid ? current_offset : -1;
							current_offset += pTrack->sample_size[sample_index];
							current_chunk_samples_number--;
							sample_index++;
						}
						if (pTrack->sample_number > sample_index)
							pTrack->sample_number = sample_index;
						if (key_sample)
						{
							for (u32 sample_index = 0; sample_index < pTrack->sample_number; sample_index++)
								pTrack->sample_type[sample_index] = kVDXVFT_Predicted;
							for (u32 key_sample_index = 0; key_sample_index < key_sample_number; key_sample_index++)
								if (key_sample[key_sample_index] < pTrack->sample_number && key_sample[key_sample_index])// >= 763)
									pTrack->sample_type[key_sample[key_sample_index]] = kVDXVFT_Independent;
							pTrack->sample_type[0] = kVDXVFT_Independent;
						}
						else
							for (u32 sample_index = 0; sample_index < pTrack->sample_number; sample_index++)
								pTrack->sample_type[sample_index] = kVDXVFT_Independent;
						if (pTrack->sample_number)
							pTrack->tread |= MP4Track::tread_stbl;
					}
		}
		if (pTrack->tread & MP4Track::tread_ctts)
		{
			if (composite_time_number < pTrack->sample_number)
			{
				i32 *p = new i32[pTrack->sample_number];
				if (p)
				{
					CopyMemory(p, pTrack->sample_composite_time, composite_time_number*4);
					ZeroMemory(p + composite_time_number*4, (pTrack->sample_number - composite_time_number)*4);
				}
				else
					pTrack->tread &= ~(MP4Track::tread_ctts | MP4Track::tread_stbl);
				delete[] pTrack->sample_composite_time;
				pTrack->sample_composite_time = p;
			}
			if (pTrack->tread & MP4Track::tread_ctts)
			{
				i64 end = pTrack->sample_time[pTrack->sample_number] + pTrack->sample_composite_time[pTrack->sample_number - 1];
				const u32 sample_number = klMin(composite_time_number, pTrack->sample_number);
				for (u32 i = 0; i < sample_number; i++)
				{
					if (end < pTrack->sample_time[i + 1] + pTrack->sample_composite_time[i])
						end = pTrack->sample_time[i + 1] + pTrack->sample_composite_time[i];
				}
				pTrack->duration = end - pTrack->sample_composite_time[0];
			}
		}
		else
		{
			pTrack->duration = pTrack->sample_time[pTrack->sample_number];
			if (pTrack->sample_composite_time = new i32[pTrack->sample_number])
				ZeroMemory(pTrack->sample_composite_time, pTrack->sample_number*4);
			else
				pTrack->tread &= ~MP4Track::tread_stbl;
		}
		if (pTrack->tread & MP4Track::tread_stbl)
			if (pTrack->frame_to_sample = new u32[pTrack->sample_number])
			{
				if (pTrack->tread & MP4Track::tread_ctts)
					if (sort_time_table *p = new sort_time_table[pTrack->sample_number])
					{
						for (u32 i = 0; i < pTrack->sample_number; i++)
						{
							p[i].time = pTrack->sample_time[i] + pTrack->sample_composite_time[i];
							p[i].index = i;
						}
						for (u32 i = 1; i < pTrack->sample_number; i++)
							for (u32 ref = i-1; i < pTrack->sample_number && p[i].time < p[ref].time && pTrack->sample_type[i] != kVDXVFT_Independent; i++)
								pTrack->sample_type[i] = kVDXVFT_Bidirectional;
						qsort(p, pTrack->sample_number, sizeof(sort_time_table), sort_time_compare);
						for (u32 i = 0; i < pTrack->sample_number; i++)
							pTrack->frame_to_sample[i] = p[i].index;
						delete[] p;
					}
					else
						pTrack->tread &= ~MP4Track::tread_stbl;
				else
					for (u32 i = 0; i < pTrack->sample_number; i++)
						pTrack->frame_to_sample[i] = i;
			}
			else
				pTrack->tread &= ~MP4Track::tread_stbl;
		if (pTrack->tread & MP4Track::tread_stbl)
			if (wcslen(mBaseFileName) < sizeof(pTrack->file_name)/sizeof(*pTrack->file_name))
				wcscpy(pTrack->file_name, mBaseFileName);
			else
				pTrack->tread &= ~MP4Track::tread_stbl;
	}
	delete[] chunk_info;
	delete[] chunk_offset;
	delete[] key_sample;
}

uint VDInputFilePluginMP4::ParseEsds(u64 pos, const u64 esds_end, u32 *bitrate_avg, u32 *codec_data_len, u64 *codec_data_pos)
{
	uint type = 0;
	u32 bitrate = 0;
	u32 len = 0;
	Seek(pos);
	if (pos + 6 <= esds_end && !(Read4() & 0xFF000000))
	{
		pos += Read1() == 3 ? ReadV(NULL) + 8 : 7;
		if (Seek(pos) && Read1() == 4)
		{
			u32 offset = ReadV(&len);
			const u64 pos2_end = klMin(pos + offset + len + 1, esds_end);
			pos += offset + 1;
			if (pos < pos2_end)
			{
				type = Read1();
				Read8();
				bitrate = Read4();
				len = 0;
				if (Read1() == 5)
				{
					offset = ReadV(&len);
					if (pos + offset + len + 0x0E > pos2_end) //    -  
						len = pos + offset + 0x0E >= pos2_end ? 0 : (u32)(pos2_end - (pos + offset + 0x0E));
					if (codec_data_pos)
						*codec_data_pos = pos + offset + 0x0E;
				}
			}
		}
	}
	if (bitrate_avg)
		*bitrate_avg = bitrate;
	if (codec_data_len)
		*codec_data_len = len;
	return type;
}

bool VDInputFilePluginMP4::ParseVideoType(u64 pos, const u64 pos_end, u32 codingname, MP4Track *pTrack)
{
	u64 vibox_pos = -1;
	u32 bitrate_avg = 0;
	u32 pixel_aspect_x = 0, pixel_aspect_y = 0;
	if (pos_end > pos && pos_end - pos >= 0x4E + 8)
	{
		u64 pos2 = pos + 0x4E;
		while (pos2 < pos_end)
		{
			MP4Box box;
			u32 offset = ReadBox(&box, pos2);
			switch (box.type)
			{
				case 'avcC':
				case 'esds':
				case 'd263':
					if (vibox_pos == -1)
						vibox_pos = pos2;
					break;
				case 'btrt':
					if (!bitrate_avg && Seek(pos2 + offset + 8))
						bitrate_avg = Read4();
					break;
				case 'pasp':
				{
					u32 x = Read4();
					u32 y = Read4();
					if (!pixel_aspect_x)
						pixel_aspect_x = x;
					if (!pixel_aspect_y)
						pixel_aspect_y = y;
				}
			}
			if ((i64)box.size >= offset)
				pos2 += box.size;
			else
				break;
		}
	}
	if (vibox_pos != -1)
	{
		MP4Box box;
		u32 offset = ReadBox(&box, vibox_pos);
		const u64 vibox_end = klMin(vibox_pos + box.size, pos_end);
		switch (box.type)
		{
			case 'avcC':
			{
				if (vibox_pos + offset + 6 <= vibox_end)
				{
					u64 posh = vibox_pos + offset + 6;
					uint codec_data_len = 0;
					uint i;
					Seek(vibox_pos + offset + 5);
					for (i = Read1() & 0x1F; i--;)
					{
						uint len = Read2() + 2;
						if (posh + len > vibox_end)
							break;
						codec_data_len += len;
						posh += len;
						Seek(posh);
					}
					if (i == -1)
						for (i = Read1(), posh++; i--;)
						{
							uint len = Read2() + 2;
							if (posh + len > vibox_end)
								break;
							codec_data_len += len;
							posh += len;
							Seek(posh);
						}

					if (FillVideoMediaType(pTrack, pos, codingname, codec_data_len, pixel_aspect_x, pixel_aspect_y))
					{
						uint lenh = 0;
						u8 *p = (u8*)(pTrack->format.video + 1);
						Seek(vibox_pos + offset + 5);
						for (uint i = Read1() & 0x1F; i--;)
						{
							uint len = Read2() + 2;
							if (lenh + len > codec_data_len)
								break;
							*(u16*)p = _byteswap_ushort(len - 2);
							p += 2;
							Read(p, len - 2);
							p += len - 2;
						}
						for (uint i = Read1(); i--;)
						{
							uint len = Read2() + 2;
							if (lenh + len > codec_data_len)
								break;
							*(u16*)p = _byteswap_ushort(len - 2);
							p += 2;
							Read(p, len - 2);
							p += len - 2;
						}
						return true;
					}
				}
				break;
			}
			case 'esds':
			{
				u32 bitrate_avg_local;
				u32 codec_data_len;
				u64 codec_data_pos;
				uint type = ParseEsds(vibox_pos + offset, vibox_end, &bitrate_avg_local, &codec_data_len, &codec_data_pos);
				if (type == 0x20 || (type >= 0x60 && type <= 0x65) || type == 0x6A) // mpeg4-2 / mpeg2-2 / mpeg1-2
					if (FillVideoMediaType(pTrack, pos, type == 0x20 ? codingname : type == 0x6A ? 'MPG1' : 'MPG2', codec_data_len, pixel_aspect_x, pixel_aspect_y))
					{
						if (codec_data_len && Seek(codec_data_pos))
							Read(pTrack->format.video + 1, codec_data_len);
						return true;
					}
				break;
			}
			case 'd263':
			{
				u32 codec_data_len = (u32)klMax<i64>(klMin(box.size, vibox_end - vibox_pos) - offset, 0);
				if (FillVideoMediaType(pTrack, pos, codingname, codec_data_len, pixel_aspect_x, pixel_aspect_y))
				{
					if (codec_data_len && Seek(vibox_pos + offset))
						Read(pTrack->format.video + 1, codec_data_len);
					return true;
				}
				break;
			}
		}
	}
	return false;
}

bool VDInputFilePluginMP4::FillVideoMediaType(MP4Track *pTrack, u64 pos, u32 codingname, u32 codec_data_len, u32 pixel_aspect_x, u32 pixel_aspect_y)
{
	pTrack->format_size = sizeof(BITMAPINFOHEADER) + codec_data_len;
	if (pTrack->format.video = (BITMAPINFOHEADER*) new u8[pTrack->format_size])
	{
		memset(pTrack->format.video, 0, pTrack->format_size);
		pTrack->format.video->biSize = sizeof(BITMAPINFOHEADER);
		Seek(pos + 0x18);
		pTrack->format.video->biWidth  = Read2();
		pTrack->format.video->biHeight = Read2();
		pTrack->format.video->biPlanes = 1;
		Seek(pos + 0x4A);
		pTrack->format.video->biBitCount = Read2();
		pTrack->format.video->biCompression = _byteswap_ulong(codingname);
		if (pixel_aspect_x && pixel_aspect_y)
		{
			pTrack->format.video->biXPelsPerMeter = pixel_aspect_y;
			pTrack->format.video->biYPelsPerMeter = pixel_aspect_x;
		}
		else
		{
			pTrack->format.video->biXPelsPerMeter = 1;
			pTrack->format.video->biYPelsPerMeter = 1;
		}
		return true;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////

class VDInputFileDriverPluginMP4 : public vdxunknown<IVDXInputFileDriver> {
public:
	VDInputFileDriverPluginMP4(const VDXInputDriverContext& context);
	~VDInputFileDriverPluginMP4();

	int		VDXAPIENTRY DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize);
	bool	VDXAPIENTRY CreateInputFile(uint32 flags, IVDXInputFile **ppFile);

protected:
	const VDXInputDriverContext& mContext;
};

VDInputFileDriverPluginMP4::VDInputFileDriverPluginMP4(const VDXInputDriverContext& context)
	: mContext(context)
{
}

VDInputFileDriverPluginMP4::~VDInputFileDriverPluginMP4() {
}

int VDXAPIENTRY VDInputFileDriverPluginMP4::DetectBySignature(const void *pHeader, sint32 nHeaderSize, const void *pFooter, sint32 nFooterSize, sint64 nFileSize) {
	return -1;
}

bool VDXAPIENTRY VDInputFileDriverPluginMP4::CreateInputFile(uint32 flags, IVDXInputFile **ppFile) {
	IVDXInputFile *p = new VDInputFilePluginMP4(mContext);
	if (!p)
		return false;

	*ppFile = p;
	p->AddRef();
	return true;
}

///////////////////////////////////////////////////////////////////////////////

bool VDXAPIENTRY mp4_create(const VDXInputDriverContext *pContext, IVDXInputFileDriver **ppDriver) {
	IVDXInputFileDriver *p = new VDInputFileDriverPluginMP4(*pContext);
	if (!p)
		return false;
	*ppDriver = p;
	p->AddRef();
	return true;
}

const uint8 mp4_sig[] = {
	  0, 0xFF,
	  0, 0xFF,
	  0, 0x00,
	  0, 0x00,
	'f', 0xFF,
	't', 0xFF,
	'y', 0xFF,
	'p', 0xFF,
};

const VDXInputDriverDefinition mp4_input={
	sizeof(VDXInputDriverDefinition),
	VDXInputDriverDefinition::kFlagSupportsVideo | VDXInputDriverDefinition::kFlagSupportsAudio,
	10,
	sizeof(mp4_sig),
	mp4_sig,
	L"*.mp4|*.3gp",
	L"MP4 files (*.mp4; *.3gp)|*.mp4;*.3gp",
	L"MP4 input driver",
	mp4_create
};

extern const VDXPluginInfo mp4_plugin={
	sizeof(VDXPluginInfo),
	L"MP4 input driver",
	L"Skakov Pavel",
	L"Loads MP4 files.",
	0x00010000,
	kVDXPluginType_Input,
	0,
	kVDXPlugin_APIVersion,
	kVDXPlugin_APIVersion,
	kVDXPlugin_InputDriverAPIVersion,
	kVDXPlugin_InputDriverAPIVersion,
	&mp4_input
};

///////////////////////////////////////////////////////////////////////////////

const VDPluginInfo *const kPlugins[]={
	&mp4_plugin,
	NULL
};

extern "C" const VDPluginInfo *const * VDXAPIENTRY VDGetPluginInfo() {
	return kPlugins;
}
