﻿using System;
using System.Windows.Forms;
using System.Diagnostics;
using System.Collections;
using Microsoft.Win32;
using System.IO;
using System.Text;
using System.Runtime.InteropServices;

namespace epgviewer
{
    public interface IMRUClient
    {
        void OpenMRUFile(string File);
    }
    public class MRUManager
    {
        #region Members
        private Form ownerForm;
        private MenuItem menuItemMRU;
        private MenuItem menuItemParent;
        private string registryPath;
        private int maxNumberOfFiles = 10;
        private int maxDisplayLength = 40;
        private string currentDirectory;
        private ArrayList mruList;
        private const string regEntryName = "file";
        #endregion

        #region Windows API
        [DllImport("shlwapi.dll", CharSet = CharSet.Auto)]
        private static extern bool PathCompactPathEx(
            StringBuilder pszOut,
            string pszPath,
            int cchMax,
            int reserved);
        #endregion

        #region Constructor
        public MRUManager()
        {
            mruList = new ArrayList();
        }
        #endregion

        #region Public Properties
        public int MaxDisplayNameLength
        {
            set
            {
                maxDisplayLength = value;
                if (maxDisplayLength < 10)
                    maxDisplayLength = 10;
            }
            get
            {
                return maxDisplayLength;
            }
        }
        public int MaxMRULength
        {
            set
            {
                maxNumberOfFiles = value;
                if (maxNumberOfFiles < 1)
                    maxNumberOfFiles = 1;

                if (mruList.Count > maxNumberOfFiles)
                    mruList.RemoveRange(maxNumberOfFiles - 1, mruList.Count - maxNumberOfFiles);
            }
            get
            {
                return maxNumberOfFiles;
            }
        }
        public string CurrentDir
        {
            set
            {
                currentDirectory = value;
            }
            get
            {
                return currentDirectory;
            }
        }
        #endregion

        #region Public Functions
        public void Initialize(Form owner, MenuItem mruItem, string regPath)
        {
            ownerForm = owner;
            if (!(owner is IMRUClient))
            {
                throw new Exception(
                    "MRUManager: Owner form doesn't implement IMRUClient interface");
            }
            menuItemMRU = mruItem;
            try
            {
                menuItemParent = (MenuItem)menuItemMRU.Parent;
            }
            catch
            {
            }
            if (menuItemParent == null)
            {
                throw new Exception(
                    "MRUManager: Cannot find parent of MRU menu item");
            }
            registryPath = regPath;
            if (registryPath.EndsWith("\\"))
                registryPath += "MRU";
            else
                registryPath += "\\MRU";
            currentDirectory = Directory.GetCurrentDirectory();
            menuItemParent.Popup += new EventHandler(this.OnMRUParentPopup);
            ownerForm.Closing += new System.ComponentModel.CancelEventHandler(OnOwnerClosing);
            LoadMRU();
        }
        public void Add(string file)
        {
            Remove(file);
            if (mruList.Count == maxNumberOfFiles)
                mruList.RemoveAt(maxNumberOfFiles - 1);
            mruList.Insert(0, file);
        }
        public void Remove(string file)
        {
            int i = 0;
            IEnumerator myEnumerator = mruList.GetEnumerator();
            while (myEnumerator.MoveNext())
            {
                if ((string)myEnumerator.Current == file)
                {
                    mruList.RemoveAt(i);
                    return;
                }
                i++;
            }
        }
        #endregion

        #region Event Handlers
        private void OnMRUParentPopup(object sender, EventArgs e)
        {
            if (menuItemMRU.IsParent)
            {
                menuItemMRU.MenuItems.Clear();
            }
            if (mruList.Count == 0)
            {
                menuItemMRU.Enabled = false;
                return;
            }
            menuItemMRU.Enabled = true;
            MenuItem item;
            IEnumerator myEnumerator = mruList.GetEnumerator();
            while (myEnumerator.MoveNext())
            {
                item = new MenuItem(GetDisplayName((string)myEnumerator.Current));
                item.Click += new EventHandler(this.OnMRUClicked);
                menuItemMRU.MenuItems.Add(item);
            }
        }
        private void OnMRUClicked(object sender, EventArgs e)
        {
            string s;
            try
            {
                MenuItem item = (MenuItem)sender;
                if (item != null)
                {
                    s = (string)mruList[item.Index];
                    if (s.Length > 0)
                    {
                        ((IMRUClient)ownerForm).OpenMRUFile(s);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine("Exception in OnMRUClicked: " + ex.Message);
            }
        }
        private void OnOwnerClosing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            int i, n;
            try
            {
                RegistryKey key = Registry.CurrentUser.CreateSubKey(registryPath);
                if (key != null)
                {
                    n = mruList.Count;
                    for (i = 0; i < maxNumberOfFiles; i++)
                    {
                        key.DeleteValue(regEntryName + i.ToString(), false);
                    }
                    for (i = 0; i < n; i++)
                    {
                        key.SetValue(regEntryName + i.ToString(), mruList[i]);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine("Saving MRU to Registry failed: " + ex.Message);
            }
        }
        #endregion

        #region Private Functions
        private void LoadMRU()
        {
            string sKey, s;
            try
            {
                mruList.Clear();
                RegistryKey key = Registry.CurrentUser.OpenSubKey(registryPath);
                if (key != null)
                {
                    for (int i = 0; i < maxNumberOfFiles; i++)
                    {
                        sKey = regEntryName + i.ToString();
                        s = (string)key.GetValue(sKey, "");
                        if (s.Length == 0)
                            break;
                        mruList.Add(s);
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine("Loading MRU from Registry failed: " + ex.Message);
            }
        }
        private string GetDisplayName(string fullName)
        {
            FileInfo fileInfo = new FileInfo(fullName);
            if (fileInfo.DirectoryName == currentDirectory)
                return GetShortDisplayName(fileInfo.Name, maxDisplayLength);
            return GetShortDisplayName(fullName, maxDisplayLength);
        }
        private string GetShortDisplayName(string longName, int maxLen)
        {
            StringBuilder pszOut = new StringBuilder(maxLen + maxLen + 2);
            if (PathCompactPathEx(pszOut, longName, maxLen, 0))
            {
                return pszOut.ToString();
            }
            else
            {
                return longName;
            }
        }
        #endregion
    }
}